#!/usr/bin/rexx
/*
   Name:    setupBSF.rex
   Purpose: set up BSF4Rexx such, that it can be used easily (attempt to serve users
            without any knowledge about Java installation issues)
   Author:  Rony G. Flatscher
   Date:    2006-01-07
   Version: 1.13
   Changes:
            2008-08-10, ---rgf, changed cmd.eCopy from "-plfv" to "-fv" to allow successful
                                execution from Ubuntu on a Windows NTFS-system;
                                on Unix: adding full path to setEnvironmentBSF.sh
            2008-07-14, ---rgf, added "canonizePath()"; changed logic for "setEnvironemnt4BSF"
                                on Unix; will now set LD_LIBRARY_PATH to point to a library
                                that has the dynamic link library included; lists it last on
                                LD_LIBRARY_PATH
            2008-05-17, ---rgf, now only runs on ooRexx due to usage of directives and OLEObject;
                                creates an install and uninstall script for BSF (adding paths to
                                PATH, CLASSPATH; if rexx.exe is put into user's PATH then HKCU is
                                used, otherwise HKLM)

            2007-02-03, ---rgf, changed logic of installation script to create a "rexxj.{cmd|sh},
                                which sets all needed environment values to run the Rexx scripts
                                from everywhere; this way it is not necessary (and not advised!)
                                to install BSF4Rexx as a Java extension (one JDBC-related error,
                                which I have not been able to trace down yet);
            2007-01-31, ---rgf, made "bsf4rexx.{cmd|sh}" simpler, hence easier to read and to
                                maintain; JavaInfo4BSF4RexxInstallation.java now also returns
                                information about BSFManager.java and RexxEngine.java, if present
                                on the system, e.g.
                                    org.apache.commons.logging.LogFactory=[1]
                                    org.apache.bsf.BSFManager.available=[1]
                                    org.apache.bsf.BSFManager.version=[242.20070128]
                                    org.rexxla.bsf.engines.rexx.RexxEngine.available=[1]
                                    org.rexxla.bsf.engines.rexx.RexxEngine.version=[208.20070128]
            2006-12-17, ---rgf, Java 6.0 introduces a second directory in "java.ext.dirs",
                                using first one ("jre/lib/ext"; script had assumed only one
                                directory)
            2006-12-08, ---rgf, since BSF 2.4.1 (20061208), commons-logging is not needed
                                anymore, hence adopting install script
            2006-01-07, ---rgf, incorporated Mark Miesfeld fix (wording, parseArgs)
                                fixed bug, if "." was missing in set CLASSPATH
            2006-01-15, ---rgf, on Windows make sure that path chunks get quotes removed
            2006-01-17, ---rgf, split jar-file into bsf-v???-yyyymmdd.jar and bsf-rexx-engine.jar,
                                take that into account for copying and removing
            2006-01-22, ---rgf, on Linux hint the user how to enter "setEnvironment4BSF4Rexx.sh"
            2006-01-24, ---rgf, make sure that support infos are quoted with apostrophes on Unix,
                                with quotes on Windows, added qc() for that purpose
            2006-02-06, ---rgf, Unix version now prepared to add setEnvironment4BSF4Rexx.sh to .bashrc
                                automatically
            2006-02-25, ---rgf, "setEnvironment4BSF4Rexx.*" is now copied to a PATH directory,
                                corrected quoting for Unix echo-outputs
            2006-08-01, ---rgf, fixed a bug which prevented from "setEnvironment4BSF4Rexx.*" to be
                                deleted upon uninstall; corrected grammar error; takes care to
                                copy "commons-logging-1.1.jar", if necessary; fixed a bug, now
                                creates working "bsf4rexx.{cmd|sh}"
            2006-09-07, ---rgf, added the directory "jrockit" ("BEA Java"), kudos for this go to
                                Mette (mettemusens .. hotmail.com)

   last svn check-in: $Revision: 274 $ $Author: rony $ $Date: 2008-08-19 12:01:14 +0200 (Tue, 19 Aug 2008) $

   Usage:   rexx setupBSF [path2java.exe [, path2binDir-to-install-Rexx-scripts-to]]
            ... "path2java.exe": optional path to java executable
            ... "path2binDir-to-install-Rexx-scripts-to": optional path to directory
                into which the supporting rexx-scripts should be copied to; defaults
                to "/usr/bin" on Linux and to the ooRexx-homedirectory on Windows

   license:

    ------------------------ Apache Version 2.0 license -------------------------
       Copyright (C) 2006-2008 Rony G. Flatscher

       Licensed under the Apache License, Version 2.0 (the "License");
       you may not use this file except in compliance with the License.
       You may obtain a copy of the License at

           http://www.apache.org/licenses/LICENSE-2.0

       Unless required by applicable law or agreed to in writing, software
       distributed under the License is distributed on an "AS IS" BASIS,
       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
       See the License for the specific language governing permissions and
       limitations under the License.
    -----------------------------------------------------------------------------
*/

cmd.=""                          /* default value for stem        */
cmd.eArgSeparator="09"x          /* separator for delimiting Rexx arguments   */
                                 /* paths to Java and desired bin-dir given?  */
parse value parseArgs(arg(1)) with cmd.eJavaBinary (cmd.eArgSeparator) cmd.eBinDir

cmd.=""                          /* default value for stem        */
call getJavaProps                /* get the properties of the installed Java  */
/*
call dumpJavaProps               /* dump the Java properties in sorted order  */
*/

/* for forcing testing Unix on Windows
j.file.separator="/" -- rgf, debug
j.path.separator=":" -- rgf, debug
*/

   /* parse path, put directories into the stem variable */
path.=""
call parsePath

if j.path.separator="" | j.file.separator="" then
do
   say "PANIC: could not use 'java' to get at its properties!"
   say
   say "Please make sure that you have Java installed! You can run 'setupBSF.rex'"
   say "with an argument being the fully qualified path to the 'java' executable"
   say "for which you wish to create the 'un/install' and 'bsf4rexx' scripts."
   say
   call usage
end

      /* determine commands depending on Unix or Windows (OS2)    */
cmd.eIsUnix=(j.file.separator="/")
      /* DLL/SO-extension library usually named "i386" on Intel, and "solaris" for Sun's Java   */
cmd.eExt.dir.bin=j.java.home || j.file.separator || "lib" || j.file.separator || ,
                                iif(pos("86", j.os.arch)>0, "i386", j.os.arch)



      /* determine CLASSPATH environment variable value  */
/* 2007-01-28, ---rgf
call SysFileTree "bsf-*.jar", "cmd.eJarBaseNames.", "FO" /* query for the jars we need */
cmd.eSetCP=iif(cmd.eIsUnix=1, "", "set ") || "CLASSPATH="
do i=1 to cmd.eJarBaseNames.0    /* iterate over all found jar-files    */
   cmd.eSetCP=cmd.eSetCP || iif(i=1, "", j.path.separator) || cmd.eJarBaseNames.i
end
*/

/* 2007-01-28, ---rgf */
   /* set BSF4REXX_HOME */
cmd.eSetBSF4REXX_HOME=iif(cmd.eIsUnix=1, "", "set ") "BSF4REXX_HOME="j.user.dir
call SysFileTree "bsf*.jar", "cmd.eJarBaseNames.", "FO" /* query for the jars we need */
cmd.eSetCP=iif(cmd.eIsUnix=1, "", "set ") || "CLASSPATH="
do i=1 to cmd.eJarBaseNames.0    /* iterate over all found jar-files    */
   cmd.eSetCP=cmd.eSetCP || iif(i=1, "", j.path.separator) || -
                            iif(cmd.eIsUnix=1, "$BSF4REXX_HOME", "%BSF4REXX_HOME%") || -
                            j.file.separator || filespec("Name", cmd.eJarBaseNames.i)
end


/* 2006-08-02, ---rgf */
     /* get and set the commons-logging jar-file name: */
   cmd.eCommonsLoggingJarName=""    /* default to empty  */
   call sysFileTree "commons-logging-*.jar", "clname.", "FO"   /* get fully qualified file name */
   if clname.0>0 then
   do
      cmd.eCommonsLoggingJarName=clname.1    /* save very first (should be only) filename */

         /* extend classpath  */
      cmd.eSetCP=cmd.eSetCP || j.path.separator || cmd.eCommonsLoggingJarName

         /* add to jar-stem   */
      cmd.eJarBaseNames.0=i         /* i got already increased by 1, so no need to do it again */
      cmd.eJarBaseNames.i=cmd.eCommonsLoggingJarName
   end
/* 2006-08-02, ---rgf */

cmd.eSetCP=cmd.eSetCP || j.path.separator || "." || j.path.separator || -
                         iif(cmd.eIsUnix=1, "$CLASSPATH", "%CLASSPATH%")


   /* define path to "jvm.dll/libjvm.so" ("javai.dll/libjavai.so" for Java 1.1.8)   */
cmd.ePath2JavaDLL= getPath2JavaDll()

cmd.eThisDirectory=directory() || j.file.separator -- save this directory

cmd.eSupportInfos=qc("Support: Internet newsgroup <news:comp.lang.rexx> or via <http://www.RexxLA.org>")



if cmd.eIsUnix then
do
   nl="0a"x                      /* Unix new-line character sequence (\n)     */
   cmd.eLeadIn="#!/bin/sh"nl     /* command to execute this file              */
   cmd.eEcho="echo"
   -- cmd.eCopy="cp -plfv"          /* preserve, hard-link, force, verbose: source target(dir)  */
   cmd.eCopy="cp -fv"            /* force, verbose: source target(dir) [works on FAT32/NTFS systems from Linux as well] */
   cmd.eLink="ln -fv"            /* force, verbose: source target(dir)  */
   cmd.eDelete="rm -fv"
   cmd.eSetPath="export PATH="                 || "$BSF4REXX_HOME"  || j.path.separator || "$PATH"
   /*
       No need to list the Java-known library paths (JVM will find it on its own)
   */
   cmd.eSetDLLPath="export LD_LIBRARY_PATH=$BSF4REXX_HOME"

   if value("LD_LIBRARY_PATH",,"ENVIRONMENT")<> "" then  -- if LD_LIBRARY_PATH is set, then use it
      cmd.eSetDLLPath = cmd.eSetDLLPath || j.path.separator || "$LD_LIBRARY_PATH"

   cmd.eSetDLLPath = cmd.eSetDLLPath || j.path.separator  || cmd.ePath2JavaDLL

   /* 2008-07-14, rgf
   cmd.eSetDLLPath="export LD_LIBRARY_PATH=$BSF4REXX_HOME"  || j.path.separator  || "$LD_LIBRARY_PATH"
   */


   cmd.eInstallScriptName="installBSF.sh"
   cmd.eUninstallScriptName="uninstallBSF.sh"
   cmd.eTestScriptName   = "rexxj.sh"  /* "testBSF4Rexx.sh"             */
   -- cmd.eSetEnvironmentScriptName="setEnvironment4BSF.sh"
   cmd.eSetEnvironmentScriptName=cmd.eThisDirectory || "setEnvironment4BSF.sh"


   cmd.eAdd2BashRc   = "~/.bashrc"   /* bash profile to add entry to        */
   cmd.eAdd2BashRc.0 = 3
   cmd.eAdd2BashRc.1 = "    # allow BSF4Rexx to find the Java dynamic library (such that Java can be demand loaded)"
   cmd.eAdd2BashRc.2 = "    ." cmd.eSetEnvironmentScriptName ">/dev/null"

   cmd.eSetExecutable    ="chmod 755"     /* set executable for all (user, group, other)  */
   cmd.eLineComment="#"
   cmd.eDLL="libBSF4Rexx.so"     /* define name of dynamic link library    */
   if cmd.eBinDir="" then
      cmd.eBinDir="/usr/bin"     /* directory where executables reside     */

   cmd.eLibDir="/usr/lib"
   /*  cmd.eQuote=''                 /* no quoting of file names for Unix      */ */
   cmd.eQuote='"'                 /* ---rgf, 2007-01-28: quoting of file names for Unix too! */
   cmd.eAllArguments='$*'
   if cmd.eJavaBinary="" then
      cmd.eJavaBinary=getFromPath("java")    /* get fully qualified path   */
end
else  /* Windows  */
do
   nl="0d0a"x                    /* Windows/OS2 new-line character sequence (\n) */
   cmd.eSetLocal="setlocal"      /* Windows command to set a local environment*/
   cmd.eEcho="@echo"
   cmd.eEchoOff="@echo off"
   cmd.eCopy="copy /y"
   cmd.eLink=cmd.eCopy           /* no ln available, hence copy   */
   cmd.eDelete="del /f"
   cmd.eSetPath="set PATH="      || "%BSF4REXX_HOME%" || j.path.separator || "%PATH%"
   cmd.eSetDLLPath="set PATH="   || cmd.ePath2JavaDLL || j.path.separator || "%PATH%"
   cmd.eInstallScriptName="installBSF.cmd"
   cmd.eUninstallScriptName="uninstallBSF.cmd"
   cmd.eTestScriptName   = "rexxj.cmd" /* "testBSF4Rexx.cmd"            */
   cmd.eSetEnvironmentScriptName="setEnvironment4BSF.cmd"
   cmd.eLineComment="REM "
   cmd.eDLL="BSF4Rexx.dll"       /* define name of dynamic link library    */
   if cmd.eBinDir="" then
      cmd.eBinDir=getRexxDir()   /* directory where executables reside     */

   cmd.eLibDir=cmd.eBinDir
   cmd.eQuote='"'                /* always quote file names for Windows    */
   cmd.eAllArguments='%*'
   if cmd.eJavaBinary="" then
      cmd.eJavaBinary=qw(getFromPath("java.exe"))   /* get fully qualified path   */

   if isRexxInstalledInHKCU() then  -- ooRexx in user hive (HKCU: current user)
   do
      cmd.eRegistryPath     ="HKCU\Environment\Path"
      cmd.eRegistryClassPath="HKCU\Environment\CLASSPATH"
   end
   else                             -- ooRexx in system hive (HKLM: local machine)
   do
      cmd.eRegistryPath     ="HKLM\SYSTEM\CurrentControlSet\Control\Session Manager\Environment\Path"
      cmd.eRegistryClassPath="HKLM\SYSTEM\CurrentControlSet\Control\Session Manager\Environment\CLASSPATH"
   end
end

if cmd.eJavaBinary="" | cmd.eJavaBinary='""' then
do
   javaName="'java"iif(cmd.eIsUnix,"",".exe")"'"
   say "PANIC: no executable" javaName "found !"
   say
   say "Please make sure that you have Java installed! You can run 'setupBSF.rex'"
   say "with an argument being the fully qualified path to the" javaName "executable"
   say "for which you wish to create the 'un/install' and 'bsf4rexx' scripts."
   say
   call usage
end


tmpStr=cmd.eJavaBinary "-Djava.library.path=" || cmd.eQuote || j.user.dir /* make sure Java finds BSF4Rexx-dll   */

/* 2007-02-03, ---rgf: seems to be no need for it!
if j.java.library.path<>"" then                 /* add defined paths as well           */
do
   if cmd.eIsUnix=0 then
      tmpStr=tmpStr || j.path.separator || j.java.library.path
   else     /* Windows? Make sure no quotes within path  */
      tmpStr=tmpStr || j.path.separator || changestr('"', j.java.library.path, '')
end
*/

tmpStr=tmpStr || cmd.eQuote

cmd.eRexxDispatch= "org.rexxla.bsf.RexxDispatcher" cmd.eAllArguments
cmd.eTestStartRexx=tmpStr cmd.eRexxDispatch

   /* create the scripts   */
call create_REXXJ_Script

call createEnvironmentScript  /* Windows probably needs this script        */


call createInstallScript      /* new on 2008-05-17, ---rgf
                                 Windows: will add environment values to registry,
                                 Linux:   will add the execution of setEnvironment4BSF4Rexx.sh to ~/bash.rc
                              */

/* 2007-02-03, ---rgf: do not create the script to install BSF4Rexx as a Java extension (yet)
                       - there is a problem with JDBC-driver-loading and the ClassLoader being used;
                         needs to be debugged (could not do it to the gory details, need more time

call createInstallScript4javaExtension   /* also creates the matching uninstall script      */
*/

call info                  /* tell user what one can do now                   */

exit


/* create the install script: this will copy the BSF4Rexx jar-file and dynamic link library
   to the appropriate Java extension directory; in effect using BSF4Rexx as a genuine extension
   to Java;
   shortcomings as of 2007-02-03:
       - problems with JDBC samples that do not work, because drivers are loaded wrongly!
       - this script needs to be created and run for every Java possessing its own directory
         structure, which may be rather error prone
*/
createInstallScript: procedure expose j. cmd. path.
  fn=cmd.eInstallScriptName            /* name of install script              */
  ui=cmd.eUninstallScriptName          /* name of uninstall script            */

  call stream fn, "c", "open replace"
  call stream ui, "c", "open replace"

  hintString="created:" q(date("S") time()) "by user" q(j.user.name)

  /* call lineout fn, cmd.eLeadIn */
  call lineout fn, cmd.eEcho qe(hintString)
  call lineout fn, cmd.eEcho qe("installing ""BSF4Rexx""...")
  call lineout fn, ""
  if cmd.eIsUnix=1 then                /* make sure rights are not fully available to the world */
  do
     call lineout fn, cmd.eLineComment "make sure rights are not fully available to the world"
     call lineout fn, "chmod 755 *.sh *.so *.CLS *.rex *.dll *.class *.REX"   /* set rwxr-xr-x */
     call lineout fn, "chmod 644 *.java"                                      /* set rw-r--r-- */
     call lineout fn, ""
  end

     /* uninstall: */
  /* call lineout ui, cmd.eLeadIn */
  call lineout ui, cmd.eEcho qe(hintString)
  call lineout ui, cmd.eEcho qe("uninstalling ""BSF4Rexx""...")
  call lineout ui, ""



  if cmd.eIsUnix=1 then          /* Unix only, append ". setEnvironment..." to .bashrc  */
  do
     call lineout fn, ""
     call lineout fn, cmd.eEcho qc("Adding entries to the end of" cmd.eAdd2BashRc "...")

     appendTo=">>" cmd.eAdd2BashRc

     call lineout fn, cmd.eEcho qc(" ") appendTo
     call lineout fn, cmd.eEcho qc("#" hintString":") appendTo

     do i=1 to cmd.eAdd2BashRc.0
        call lineout fn, cmd.eEcho qc(cmd.eAdd2BashRc.i) appendTo
     end
     call lineout fn, ""
  end
  else      -- add statements to incorporate paths into registry
  do
      -- j.user.dir == %BSF4REXX_HOME%
      cmd="call rexx orx2reg.rex"      -- invoke utility to interact with Windows registry

      hint="Creating a backup of PATH and CLASSPATH registry values..."
      call lineout fn, cmd.eEcho qc(hint)
      call lineout ui, cmd.eEcho qc(hint)

      bkpFile="registryValues-"||date("S")"T"time("N")~changestr(":","")
      call lineout fn, cmd ">" bkpFile"-beforeAddBSF4RexxPathValues.bat"
      call lineout ui, cmd ">" bkpFile"-beforeDelBSF4RexxPathValues.bat"

      call lineout ui, ""
      call lineout fn, ""

         -- add current directory to PATH
      paths=j.user.dir||j.path.separator||cmd.ePath2JavaDLL

      arg="/key:"q(cmd.eRegistryPath) "/path:"q(paths)
      tmpStr1=cmd "/add"    arg
      tmpStr2=cmd "/remove" arg
         -- inform user
      call lineout fn, cmd.eEcho qc(tmpStr1)
      call lineout ui, cmd.eEcho qc(tmpStr2)

         -- carry out operation
      call lineout fn, tmpStr1
      call lineout ui, tmpStr2

      call lineout ui, ""
      call lineout fn, ""

         -- add bsf-jars to CLASSPATH
      tmpArgs=""
      do i=1 to cmd.eJarBaseNames.0
         if tmpArgs<>"" then tmpArgs=tmpArgs || j.path.separator
         tmpArgs=tmpArgs || cmd.eJarBaseNames.i
      end

      if tmpArgs<>"" then
      do
         tmpArgs=tmpArgs || j.path.separator || "."   -- add current directory (for datergf-BSF4Rexx-samples to work)

            -- make sure the current directory is searched as well
         arg="/key:"q(cmd.eRegistryClassPath) "/path:"q(tmpArgs)

         tmpStr1=cmd "/add"    arg
         tmpStr2=cmd "/remove" arg
            -- inform user
         call lineout fn, cmd.eEcho qc(tmpStr1)
         call lineout ui, cmd.eEcho qc(tmpStr2)

            -- carry out operation
         call lineout fn, tmpStr1
         call lineout ui, tmpStr2
      end
  end


  call lineout fn, ""
  call lineout fn, cmd.eEcho qe("done.")
  call lineout fn, ""
  call lineout fn, cmd.eEcho cmd.eSupportInfos
  call lineout fn, ""

  call lineout ui, ""
  call lineout ui, cmd.eEcho qe("done.")
  call lineout ui, ""
  call lineout ui, cmd.eEcho cmd.eSupportInfos
  call lineout ui, ""

  call stream fn, "c", "close"
  call stream ui, "c", "close"

  if cmd.eSetExecutable<>"" then       /* set file executable (Unix)    */
  do
     cmd.eSetExecutable cmd.eInstallScriptName
     cmd.eSetExecutable cmd.eUninstallScriptName
  end
  return


/* create the install script: this will copy the BSF4Rexx jar-file and dynamic link library
   to the appropriate Java extension directory; in effect using BSF4Rexx as a genuine extension
   to Java;
   shortcomings as of 2007-02-03:
       - problems with JDBC samples that do not work, because drivers are loaded wrongly!
       - this script needs to be created and run for every Java possessing its own directory
         structure, which may be rather error prone
   ---> not used anymore (2007-02-03)
*/
createInstallScript4javaExtension: procedure expose j. cmd. path.
  fn=cmd.eInstallScriptName            /* name of install script              */
  ui=cmd.eUninstallScriptName          /* name of uninstall script            */

  call stream fn, "c", "open replace"
  call stream ui, "c", "open replace"

  hintString="created:" q(date("S") time()) "by user" q(j.user.name)

  /* call lineout fn, cmd.eLeadIn */
  call lineout fn, cmd.eEcho qe(hintString)
  call lineout fn, cmd.eEcho qe("installing optional Java extension ""BSF4Rexx""...")
  call lineout fn, ""
  if cmd.eIsUnix=1 then                /* make sure rights are not fully available to the world */
  do
     call lineout fn, cmd.eLineComment "make sure rights are not fully available to the world"
     call lineout fn, "chmod 755 *.sh *.so *.CLS *.rex *.dll *.class *.REX"   /* set rwxr-xr-x */
     call lineout fn, "chmod 644 *.java"                                      /* set rw-r--r-- */
     call lineout fn, ""
  end

     /* uninstall: */
  /* call lineout ui, cmd.eLeadIn */
  call lineout ui, cmd.eEcho qe("created:" q(date("S") time()) "by user" q(j.user.name))
  call lineout ui, cmd.eEcho qe("uninstalling optional Java extension ""BSF4Rexx""...")
  call lineout ui, ""


  /* tgtDir=j.java.ext.dirs */
  tgtDir=getDir(j.java.ext.dirs, 1) /* since Java 6: 1st directory "jre/lib/ext", 2nd directory: system-wide */
  call lineout fn, cmd.eEcho qe("copying the BSF4Rexx jar file(s) to the Java extension directory" q(tgtDir) "...")
  call lineout ui, cmd.eEcho qe("deleting the BSF4Rexx jar file(s) from the Java extension directory" q(tgtDir) "...")

  /* call lineout fn, cmd.eCopy qw(cmd.eJar) qw(tgtDir) */
  /* call lineout ui, cmd.eDelete qw(tgtDir || j.file.separator || cmd.eJarBaseName) */
  do i=1 to cmd.eJarBaseNames.0    /* iterate over all found jar-files    */
     call lineout fn, cmd.eCopy   qw(cmd.eJarBaseNames.i) qw(tgtDir)
     call lineout ui, cmd.edelete qw(tgtDir || j.file.separator || filespec("Name", cmd.eJarBaseNames.i) )
  end

  call lineout fn, ""
  call lineout ui, ""

  /* do we need to supply the commons.logging jar ?   */
  if j.org.apache.commons.logging.LogFactory=0 then
  do
     call lineout fn, cmd.eEcho qe("/// trying to install optional org.apache.commons.logging ...")
/*
     call lineout fn, cmd.eEcho qe("NEED for org.apache.commons.logging, hence copying the file ...")
     shortName=filespec("Name", cmd.eCommonsLoggingJarName)

     call lineout fn, cmd.eEcho qe("copying the commons-logging jar" q(shortName)  "to the Java extension directory" q(tgtDir) "...")
     call lineout ui, cmd.eEcho qe("deleting the commons-logging jar" q(shortName) "from the Java extension directory" q(tgtDir) "...")
*/

     if cmd.eCommonsLoggingJarName="" then   /* hmm, not found in current directory!   */
     do
        msg="/// COULD NOT FIND the OPTIONAL jar-file for the package org.apache.commons.logging AT INSTALL TIME !"
        call lineout fn, cmd.eEcho qe(msg)
        call lineout fn, cmd.eEcho qe("///  Look for the commons logging jar-file at http://jakarta.apache.org/commons")
        call lineout fn, cmd.eEcho qe("///  and copy the jar-file to" qw(tgtDir) "yourself, if you want")
        call lineout ui, cmd.eEcho qe(msg)
     end
/*
     else   /* create the copy/delete statement */
     do
        call lineout fn, cmd.eCopy   qw(cmd.eCommonsLoggingJarName) qw(tgtDir)
        call lineout ui, cmd.edelete qw(tgtDir || j.file.separator || filespec("Name", cmd.eCommonsLoggingJarName) )
     end
*/
     call lineout fn, ""
     call lineout ui, ""
  end



  tgtDir=cmd.eExt.dir.bin
  call lineout fn, cmd.eEcho qe("copying the BSF4Rexx dynamic link library file to its Java extension directory" q(tgtDir) "...")
  call lineout fn, cmd.eCopy qw(cmd.eDLL) qw(tgtDir)
  call lineout fn, ""
     /* uninstall: */
  call lineout ui, cmd.eEcho qe("delete the BSF4Rexx dynamic link library file from its Java extension directory" q(tgtDir) "...")
  call lineout ui, cmd.eDelete qw(tgtDir || j.file.separator || cmd.eDLL)
  call lineout ui, ""


  tgtDir=cmd.eLibDir
  call lineout fn, cmd.eEcho qe("copying the BSF4Rexx dynamic link library file to the lib directory" q(tgtDir) "...")
  call lineout fn, cmd.eCopy qw(cmd.eDLL) qw(tgtDir)
  call lineout fn, ""
     /* uninstall: */
  call lineout ui, cmd.eEcho qe("delete the BSF4Rexx dynamic link library file from the lib directory" q(tgtDir) "...")
  call lineout ui, cmd.eDelete qw(tgtDir || j.file.separator || cmd.eDLL)
  call lineout ui, ""


  tgtDir=cmd.eBinDir
  call lineout fn, cmd.eEcho qe("copying the Rexx scripts and ooRexx class files to" q(tgtDir) "...")
     /* uninstall: */
  call lineout ui, cmd.eEcho qe("deleting the Rexx scripts and ooRexx class files from" q(tgtDir) "...")
  list="*.rex *.REX *.cls *.CLS *.sh *.cmd"     /* list of files to copy/uninstall  */
  processed.=""
  do i=1 to words(list)
     call sysFileTree word(list, i), "files.", "FO"    /* get fully qualified files only  */
     do k=1 to files.0           /* now copy/delete file-by-file     */
        if cmd.eIsUnix=0 then    /* if Windows, then case-insensitive file-names, do not process them twice */
        do
           w=translate(files.k)  /* turn into uppercase              */
           if processed.w<>"" then iterate   /* already processed    */
           processed.w=files.k   /* remember that we processed this  */
        end

        call lineout fn, cmd.eCopy   qw(files.k) qw(tgtDir)   /* install    */
        call lineout ui, cmd.eDelete qw(makeFileName(tgtDir, j.file.separator, files.k))  /* uninstall  */
     end
  end


/* 2006-08-01: already handled in the above loop!
   /* copying setEnvironment4* to a PATH-directory */
  tgtDir=cmd.eBinDir
  -- tgtFile=stream(cmd.eSetEnvironmentScriptName, "C","Query EXISTS")
  tgtFile=cmd.eSetEnvironmentScriptName
  call lineout fn, cmd.eEcho qe("copying"  q(tgtFile) "to" q(tgtDir) "...")
  call lineout fn, cmd.eCopy qw(tgtFile) qw(tgtDir)
  call lineout fn, ""

  call lineout ui, cmd.eEcho qe("deleting" q(tgtFile)  "from directory" q(tgtDir) "...")
  call lineout ui, cmd.eDelete qw(tgtDir || j.file.separator || tgtFile)
  call lineout ui, ""
*/



/*
  if cmd.eIsUnix=1 then          /* Unix only, append ". setEnvironment..." to .bashrc  */
  do
     call lineout fn, ""
     call lineout fn, cmd.eEcho qc("Adding entries to the end of" cmd.eAdd2BashRc "...")

     appendTo=">>" cmd.eAdd2BashRc

     call lineout fn, cmd.eEcho qc(" ") appendTo
     call lineout fn, cmd.eEcho qc("#" hintString":") appendTo

     do i=1 to cmd.eAdd2BashRc.0
        call lineout fn, cmd.eEcho qc(cmd.eAdd2BashRc.i) appendTo
     end
     call lineout fn, ""
  end
*/


  call lineout fn, ""
  call lineout fn, cmd.eEcho qe("done.")
  call lineout fn, ""
  call lineout fn, cmd.eEcho cmd.eSupportInfos
  call lineout fn, ""

  call lineout ui, ""
  call lineout ui, cmd.eEcho qe("done.")
  call lineout ui, ""
  call lineout ui, cmd.eEcho cmd.eSupportInfos
  call lineout ui, ""

  call stream fn, "c", "close"
  call stream ui, "c", "close"

  if cmd.eSetExecutable<>"" then       /* set file executable (Unix)    */
  do
     cmd.eSetExecutable cmd.eInstallScriptName
     cmd.eSetExecutable cmd.eUninstallScriptName
  end
  return




/* create the "rexxj.{cmd|sh}script which sets up all needed parameters to run Rexx programs from
   this directory */
create_REXXJ_Script: procedure expose j. cmd. path.
  fn=cmd.eTestScriptName

  call stream fn, "c", "open replace"
  call lineout fn, cmd.eEchoOff
  /* -- call lineout fn, cmd.eLeadIn */
  call lineout fn, cmd.eLineComment "created:" q(date("S") time()) "by user" q(j.user.name)
  call lineout fn, cmd.eLineComment "script to run ""BSF4Rexx"" without installing it as a Java extension..."
  call lineout fn, ""
  call lineout fn, cmd.eSetLocal       /* on Windows: create a local environment */
  call lineout fn, ""

  call lineout fn, cmd.eLineComment "set BSF4REXX_HOME variable"
  call lineout fn, cmd.eSetBSF4REXX_HOME  /* set BSF4REXX_HOME */
  call lineout fn, ""

  call lineout fn, cmd.eLineComment "set CLASSPATH environment variable"
  call lineout fn, cmd.eSetCP          /* set CLASSPATH        */
  call lineout fn, ""


  call lineout fn, cmd.eLineComment "Java start command, allows passing arguments to Rexx script"
  tmpStr1="-Djava.library.path=" || cmd.eQuote

  if cmd.eIsUnix=1 then
  do
     tmpStr1=tmpStr1 || "$BSF4REXX_HOME"  || j.path.separator || "$LD_LIBRARY_PATH" || cmd.eQuote
  end
  else
  do
     /*
     tmpStr1=tmpStr1 || "%BSF4REXX_HOME%" || j.path.separator || "%PATH%"           || cmd.eQuote
     */
     tmpStr1=tmpStr1 || "%BSF4REXX_HOME%"                                           || cmd.eQuote
  end

  tmpStr2="-classpath" cmd.eQuote || iif(cmd.eIsUnix=1, "$CLASSPATH", "%CLASSPATH%") || cmd.eQuote
  call lineout fn, cmd.eJavaBinary tmpStr1 tmpStr2 cmd.eRexxDispatch

/*
  call lineout fn, ""
  call lineout fn, cmd.eEcho qe("done.")
  call lineout fn, ""
  call lineout fn, cmd.eEcho cmd.eSupportInfos
  call lineout fn, ""

  call lineout fn, cmd.eEcho qe("usage:" fn "rexx-file-name optional_arguments")
*/

  call stream fn, "c", "close"

  if cmd.eSetExecutable<>"" then       /* set file executable (Unix)    */
     cmd.eSetExecutable cmd.eTestScriptName

  return


/* ---rgf, 2007-01-28
/* create the test script which sets up all needed parameters to run Rexx programs from
   this directory */
create_REXXJ_Script: procedure expose j. cmd. path.
  fn=cmd.eTestScriptName

  call stream fn, "c", "open replace"
  call lineout fn, cmd.eEchoOff
  call lineout fn, cmd.eLeadIn
  call lineout fn, cmd.eSetLocal       /* on Windows: create a local environment */
  call lineout fn, ""
  call lineout fn, cmd.eLineComment "created:" q(date("S") time()) "by user" q(j.user.name)
  call lineout fn, ""
  call lineout fn, cmd.eLineComment "script to run ""BSF4Rexx"" without installing it as a Java extension..."
  call lineout fn, ""

  call lineout fn, cmd.eLineComment "set CLASSPATH environment variable"
  call lineout fn, cmd.eSetCP          /* set CLASSPATH        */
  call lineout fn, ""

  call lineout fn, cmd.eLineComment "set PATH environment variable"
  call lineout fn, cmd.eSetPath        /* set PATH             */
  call lineout fn, ""

  call lineout fn, cmd.eLineComment "Java start command, allows passing arguments to Rexx script"
  call lineout fn, cmd.eTestStartRexx  /* Java start command   */
  call lineout fn, ""
  call lineout fn, cmd.eEcho qe("done.")
  call lineout fn, ""
  call lineout fn, cmd.eEcho cmd.eSupportInfos
  call lineout fn, ""

  call lineout fn, cmd.eEcho qe("usage:" fn "rexx-file-name optional_arguments")

  call stream fn, "c", "close"

  if cmd.eSetExecutable<>"" then       /* set file executable (Unix)    */
     cmd.eSetExecutable cmd.eTestScriptName

  return
*/


/* creates the script that sets up the environment for Rexx to find the Java-DLL for
   demand loading it. */
createEnvironmentScript: procedure expose j. cmd. path.
  fn=cmd.eSetEnvironmentScriptName

  call stream fn, "c", "open replace"
  /* call lineout fn, cmd.eLeadIn */
  /* call lineout fn, "" */
  call lineout fn, cmd.eLineComment "created:" q(date("S") time())  "by user" q(j.user.name)
  call lineout fn, ""
  call lineout fn, cmd.eSetBSF4REXX_HOME
  call lineout fn, ""

  call lineout fn, cmd.eLineComment "just run this script before invoking a Rexx program that loads Java on its own."
  call lineout fn, ""

  call lineout fn, cmd.eEcho qe("Setting the CLASSPATH environment variable:")
  a=iif(cmd.eIsUnix=1, "export ", "") || cmd.eSetCP || j.path.separator
  call lineout fn, cmd.eEcho "command:" qe(a)
  call lineout fn, a
  call lineout fn, ""


  if cmd.eIsUnix=1 then
  do
/* rgf, 2008-07-14: don't link anymore, could be that users do not want this at all

     call lineout fn, "if ! [ -f /usr/lib/libjvm.so ] "
     call lineout fn, "then"
     call lineout fn, "   ln -sf" cmd.ePath2JavaDLL || "/libjvm.so"  "/usr/lib/libjvm.so"
     call lineout fn, "  " cmd.eEcho qe( q(cmd.ePath2JavaDLL || "/libjvm.so") "linked symbolically to" q("/usr/lib/libjvm.so") )
     call lineout fn, "else "
     call lineout fn, "  " cmd.eEcho qe( q("libjvm.so") "already available in" q("/usr/lib")", creation of a symbolic link not carried out!" )
     call lineout fn, "  " cmd.eEcho qe("to force new links to a specific Java version use:" q("rexx setupJava.rex"))
     call lineout fn, "fi   "
*/

/*
     call lineout fn, cmd.eEcho qe(" ")
     call lineout fn, cmd.eEcho qe("For non-Java programs to be able to successfully invoke Java via the")
     call lineout fn, cmd.eEcho qe( q('libjvm') "dynamic link library the following paths to Java directories")
     call lineout fn, cmd.eEcho qe("must be set (Java property" q('java.library.path') "on this system to:")
     call lineout fn, cmd.eEcho qe("   " pp(j.java.library.path))
     call lineout fn, cmd.eEcho qe(" ")

     call lineout fn, cmd.eEcho qe("Use the following commands on the commandline:")
     call lineout fn, cmd.eEcho qe(" ")
*/
     call lineout fn, cmd.eLineComment "set LD_LIBRARY_PATH to point to the" q('libjvm.so') "dynamic library"



/*
     a=cmd.eSetDLLPath || j.path.separator || j.java.library.path
     call lineout fn, cmd.eEcho qe(a)
     call lineout fn, a
*/

     call lineout fn, cmd.eEcho qe(cmd.eSetDLLPath)
     call lineout fn, cmd.eSetDLLPath
     call lineout fn, ""

     call lineout fn, cmd.eLineComment "set PATH environment variable to point to the bsf4Rexx directory"
     call lineout fn, cmd.eEcho qe(cmd.eSetPath)
     call lineout fn, cmd.eSetPath        /* set PATH             */
     call lineout fn, ""

     call lineout fn, cmd.eEcho qe(" ")
     call lineout fn, cmd.eEcho qe("or run this script like" q(". ./"cmd.eSetEnvironmentScriptName) -
                                   "to keep the changes to the environment.")
     call lineout fn, cmd.eEcho qe(" ")
     call lineout fn, cmd.eEcho qe("It is advised to add these environment settings to the profile of your system,")
     call lineout fn, cmd.eEcho qe("e.g. add the following entries to your" q('~/.bashrc')":")
     call lineout fn, cmd.eEcho qe(" ")
     call lineout fn, cmd.eEcho qe("   # allow ooRexx and other programs to find Java ("q('libjvm.so')")")
     call lineout fn, cmd.eEcho qe("   ." cmd.eSetEnvironmentScriptName ">/dev/null")
     call lineout fn, cmd.eEcho qe(" ")
  end
  else      /* non-Unix */
  do

     call lineout fn, cmd.eEcho qe("setting the environment variable to point to the directory in which")
     call lineout fn, cmd.eEcho qe("""libjava.so"" (Linux) / ""java.dll"" (Windows) resides. This allows")
     call lineout fn, cmd.eEcho qe("Rexx programs to load Java on their own, if needed.")
     call lineout fn, ""

     call lineout fn, cmd.eSetDLLPath
     call lineout fn, ""

     call lineout fn, cmd.eLineComment "set PATH environment variable to point to the bsf4Rexx directory"
     call lineout fn, cmd.eSetPath        /* set PATH             */
     call lineout fn, ""

     call lineout fn, cmd.eEcho qe(".")
     call lineout fn, cmd.eEcho qe("It is advised to add these environment settings to your system's profile!")

     call lineout fn, cmd.eEcho qe("(This way it is always assured, that Rexx can find Java.)")
  end

  call lineout fn, ""
  call lineout fn, cmd.eEcho qe(".")
  call lineout fn, cmd.eEcho cmd.eSupportInfos
  call lineout fn, ""

  call stream fn, "c", "close"

  if cmd.eSetExecutable<>"" then       /* set file executable (Unix)    */
     cmd.eSetExecutable cmd.eSetEnvironmentScriptName

  return


/* create stem variable "sort." from "j." and sort it by property keys ascendingly */
dumpJavaProps: procedure expose j. cmd. path. sort.
   sort.=""
   max=0
   do i=1 to j.0
      sort.i=j.i.eName j.i.eValue
      max=max(max, length(j.i.eName))
   end
   max=max+2                     /* account for square brackets   */

   sort.0=j.0                    /* save total items in stem      */

   call sysStemSort "sort."      /* sort the stem                 */
   do i=1 to sort.0
      parse var sort.i n1 v1
      say right(i, 2) "-" left(pp(n1), max)":" pp(v1)
   end
   return


/* execute the Java program "JavaInfo4BSF4RexxInstallation.class" to queue all Java properties to
   the system Rexx queue and make them available via the stem variable "j."
*/
getJavaProps: procedure expose j. cmd. path.
   signal on any name usage                           /* trap to usage label     */
         /* get the Java properties    */
   if cmd.eJavaBinary="" then java="java"             /* default     */
                         else java=cmd.eJavaBinary    /* use command-line value  */

   java "-cp . JavaInfo4BSF4RexxInstallation | rxqueue"

   i=0
   max=0
   j.=""
   sort.=""
   if queued()=0 then
   do
      say "PANIC: nothing in external Rexx queue, cannot get at the Java properties, aborting..."
      say
      call usage
   end
   else
   do i=1 to 99999 while queued()<>0
      parse pull pName "=[" pVal "]"
      if pname="" then iterate      /* blank line (the last line?)   */
      max=max(max, length(pName))   /* memorize longest name         */

      parse var pVal v1 "0x" v2     /* hex-encoded string ?          */
      j.i.eName=pName               /* use a number based index      */
      j.i.eValue=v1||x2c(v2)

      j.pName=j.i.eValue            /* save in mixed case            */

      uName=translate(pName)        /* save in mixed case            */
      j.uName=j.i.eValue
      j.0=i                         /* save maximum number of entries*/
   end
   return



/* Get the fully qualified path to "jvm.dll", if multiple places, then
   prefer "client", "hotspot", "classic" (seems to be the only one), "server".
*/
getPath2JavaDll: procedure expose j. cmd. path.
   tmpStr=j.java.home

   parts="client hotspot classic server jrockit"  /* look in these directories  */

      /* removed search for [lib]javai.{so|dll}, as Java 1.1.8 is not supported anymore */
   if cmd.eIsUnix=1 then            /* Unix           */
      dlls="libjvm.so"
   else                             /* Windows, OS2   */
      dlls ="jvm.dll"

   countWords=words(dlls)
   do i=1 to countWords
      tmpFileName=j.java.home || j.file.separator || word(dlls, i)
      say "Attempt #" i "of" countWords "searching file: ["word(dlls, i)"] in ["j.java.home || j.file.separator || -
          "] or in one of its subdirectories..."
      call sysFileTree tmpFileName, "files.", "FOS"

      if files.0>0 then             /* files found?   */
      do
         do k=1 to words(parts)
            p=j.file.separator || word(parts, k) || j.file.separator    /* get word       */
            do m=1 to files.0       /* iterate over files found   */
               if pos(p, files.m)>0 then
               do
                  return substr(files.m, 1, lastpos(j.file.separator, files.m)-1)
               end
            end
         end
      end
   end

      /* default, even if directory does not exist to hint the user where to look for  */
   return tmpStr || j.file.separator || "bin" || j.file.separator || "client"


/* Parse commandline arguments, if any.
*/
parseArgs: procedure expose j. cmd. path.
   parse arg tmpStr
   if tmpStr="" then return ""      /* no arguments available        */
   tmpStr=strip(tmpStr)             /* remove leading & trailing ws  */
   if left(tmpStr,1)='"' then       /* argument enquoted             */
   do
      parse var tmpStr '"' path2javaExe '"' tmpStr
      path2javaExe='"' || path2javaExe || '"'
   end
   else
     parse var tmpStr path2javaExe tmpStr

   path2binDir=strip(strip(tmpStr), 'B', '"')   /* could be quoted on Windows (but not "apostrophed"!)   */
   binDir=path2binDir

   curDir=directory( )              /* save current directory        */
   newDir=directory( binDir )       /* See if directory exists       */
   if newDir \== binDir then
     signal usage
   z = directory( curDir )          /* restore current directory     */

   return path2javaExe || cmd.eArgSeparator || path2binDir


/*
   parse arg tmpStr
   if tmpStr="" then return ""      /* no arguments available        */
   tmpStr=strip(tmpStr)             /* remove leading & trailing ws  */
   if left(tmpStr,1)='"' then       /* argument enquoted             */
   do
      parse var tmpStr '"' path2javaExe '"' tmpStr
      path2javaExe='"' || path2javaExe || '"'
   end
   else
     parse var tmpStr path2javaExe tmpStr

   path2binDir=strip(tmpStr)

   return path2javaExe || cmd.eArgSeparator || path2binDir
*/



/* Get path to Rexx or Regina from PATH environment variable and return it;
   Windows/OS2
*/
getRexxDir: procedure expose j. cmd. path.
     /* use this precedence order to locate a path where to copy Rexx-files to  */
  dir=findPath("\ooREXX \regina rexx")
  if dir<>"" then return dir

  eyeCatcher="?!no-rexx-path-found!?"
  say "PROBLEM: could not find explicit path to a Rexx-directory!"
  say "         Please adjust the created scripts to point to any one directory listed"
  say "         with the environment variable ""PATH"", ie. replace" pp(eyecatcher) "with"
  say "         any of the listed PATH directory, and then run the scripts."
  return eyecatcher              /* indicate that not found       */


/* Look for filename in path, if found return fully qualified path.
*/
getFromPath: procedure expose j. cmd. path.
  parse arg filename

  do i=0 to path.0      /* iterate over path directories          */
     fn=stream(path.i || j.file.separator || filename, "C", "QUERY EXISTS")

     if fn<>"" then return fn /* found, return it                 */
  end
  return filename



/* Extract filename from fully qualified path in "file" and add it to "tgtDir".
*/
makeFileName: procedure
  parse arg tgtDir, sep, file
  lpos=lastpos(sep, file)
  if lpos>0 then file=substr(file, lpos+1)
  return tgtDir || sep || file

pp: procedure                    /* poor man's "pretty" print     */
  return "[" || arg(1) || "]"


q: procedure                     /* enclose string into apostrophes  */
  return '"' || arg(1) || '"'


   /* "qe": quote echo string: enquote entire string on Unix only */
qe: procedure expose j. cmd. path.
  if  cmd.eIsUnix=1 then return "'" || arg(1) || "'"  /* use apostrophes only, rgf 2006-01-18   */
                    else return arg(1)


   /* "qe": quote fully-qualified paths in Windows as they may contain a blank */
qw: procedure expose j. cmd. path.
  if  cmd.eIsUnix=0 then return """" || arg(1) || """"
                    else return arg(1)

   /* "qc": quote string in apostrophes, if Unix, in quotes, if Windows (for support infos that contain <, >) */
qc: procedure expose j. cmd. path.
  if  cmd.eIsUnix=0 then return '"' || arg(1) || '"'     /* Windows  */
                    else return "'"  || arg(1) || "'"    /* Unix     */


iif: procedure                   /* VBasic-like function          */
  if arg(1)=1 then return arg(2)
  return arg(3)

info:                            /* tell user what has happened   */
  say
  say "the following scripts have been created:"
  say
  say " " pp(cmd.eTestScriptName)": script to run ""BSF4Rexx"" without installing it as a Java extension"
  say
  if cmd.eIsUnix=0 then
  do
     say "       usage:" cmd.eTestScriptName "rexx-file optional-arguments"
     say "     example:" cmd.eTestScriptName "infoBSF.rex"
  end
  else
  do
     say "       usage:" q("./"cmd.eTestScriptName "rexx-file optional-arguments")
     say "     example:" q("./"cmd.eTestScriptName "infoBSF.rex")
  end

  say
  say
  say " " pp(cmd.eInstallScriptName)": install" q("BSF4Rexx") "and set environment variables permanently"
  say
  if cmd.eIsUnix=0 then
     say "       usage:" cmd.eInstallScriptName
  else
     say "       usage:" q("./"cmd.eInstallScriptName)

  say
  say "     remarks: - You may need to run as user ""root"" (Linux) or ""Administrator"" (Windows)."
  say "                Under Vista, you may need to run the un/install scripts with elevated privileges."
  say
  say "              - After installation you may start any Rexx-program by issuing:"
  say
  if cmd.eIsUnix=0 then
     say "                  " cmd.eSetEnvironmentScriptName

  say "                   rexx rexx-file optional-arguments"
  say
  say "                or (note the trailing ""j"", which means via ""Java""):"
  say

  if cmd.eIsUnix=0 then
     say "                   rexxj rexx-file optional-arguments"
  else
     say "                   rexxj.sh rexx-file optional-arguments"

  say
  say " " pp(cmd.eSetEnvironmentScriptName)": allows Rexx to find Java ("q("jvm.dll")") on its own"

  if cmd.eIsUnix=1 then
     say "                it is advised to run this script right after installation of ""BSF4Rexx"""
  else
     say "                it is advised to add this environment setting to your system's profile"
  say

  say " " pp(cmd.eUninstallScriptName)": use to uninstall" q("BSF4Rexx") "(removes BSF4Rexx-paths from Windows registry)"
  say
  if cmd.eIsUnix=0 then
     say "       usage:" cmd.eUninstallScriptName
  else
     say "       usage:" q(cmd.eUninstallScriptName)

  say
  say "     remarks: - You may need to run as user ""root"" (Linux) or ""Administrator"" (Windows)."
  say "                Under Vista, you may need to run the un/install scripts with elevated privileges."
  say
  say "Good luck! :-)  [You may also want to take a look at these generated scripts.]"
  exit 0

/* give user feedback and show usage of this script */
usage:
  say
  say "This program creates scripts to test, install and uninstall ""BSF4Rexx""."
  say
  say "Usage:     rexx setupBSF [fully-qualified-file-name-of-java.exe [dir-to-copy-scripts-to]]"
  say "  (dir-to-copy-scripts-to MUST exist)"
  say
  say "Examples:  rexx setupBSF"

  if cmd.eIsUnix=0 then
  do
     say "           - will use /usr/bin/java"
     say "             if no java binary is found, then execute ""rexx setupJava.rex"", which"
     say "             will create the necessary symbolic links to an installed Java."
     say "           - will copy the BSF4Rexx-support-files to /usr/bin"
     say
  end
  else
  do
     say "           - will use any ""java.exe"" found in one of the directories of the"
     say "             environment variable ""PATH"""
     say "           - will copy the BSF4Rexx-support-files to the Rexx/Regina directory, "
     say "             if found in the environment variable ""PATH"""
  end

  say
  if cmd.eIsUnix=0 then
  do
     say "           rexx setupBSF E:\jdk1.5\jre\bin\java.exe D:\windows\system32"
     say "           - will use ""java.exe"" from the given first argument"
     say "           - will copy the BSF4Rexx-support-files to the Windows system directory"
  end
  else
  do
     say "           rexx setupBSF /usr/java/j2sdk1.4.2_10/jre/bin/java /usr/bin"
     say "           - will use the binary ""java"" from the given first argument"
     say "           - will copy the BSF4Rexx-support-files to /usr/bin"
  end

  exit -1


/* parse path into stem path. */
parsePath: procedure expose j. cmd. path.
/*
say "... j.path.separator=["j.path.separator"]"
say "... j.file.separator=["j.file.separator"]"
*/

  p=value("PATH", , "ENVIRONMENT")
  i=0
  do while p<>""
     parse var p chunk (j.path.separator) p
     i=i+1

     if cmd.eIsUnix=1 then
        path.i=chunk
     else
        path.i=changestr('"', chunk, "")  /* remove any quotes from any part of the Windows path */

     path.i.upperCase=translate(chunk)  /* uppercase version */
/*
say "... i="i "chunk="pp(chunk)
*/

     path.0=i
  end

  return

/* find needle in path and return it   */
findPath: procedure expose j. cmd. path.
   parse upper arg needles    /* blank delimited, translated into uppercase   */
   do i=1 to words(needles)   /* iterate over needles                         */
      n=word(needles, i)
      do k=1 to path.0        /* iterate over paths      */
         if pos(word(needles,i), path.k.upperCase)>0 then
            return path.k
      end
   end
   return ""


/* Returns the directory as indicated by the second, optional argument (if omitted,
   very first directory is assumed); if given number is higher than available directories,
   then the last directory is returned */
getDir: procedure expose j. cmd. path.
   parse arg dirs, number
   if number="" then number=1    /* default to very first listed directory */
   tmpStr=""
   do i=1 to number while dirs<>""
      parse var dirs tmpStr (j.path.separator) dirs
   end
   return tmpStr



/* Returns the received path in canonized form, i.e. removes "." and "..", wherever possible */
canonizePath: procedure expose j. cmd. path.
   parse arg p

   tmpArr=p~makeArray(j.file.separator)
   newArr=.array~new

   k=0      -- index into new array
   do i=1 to tmpArr~items
      w=tmpArr[i]
      if w="." then iterate      -- no need to keep "current directory"
      if w="..", k>0 then        -- can we remove super directory (not if it is ".." already!)
      do
         if newArr[k]<>".." then -- a directory, removing it would be o.k.
         do
               -- on Windows make sure first word is not a drive letter! (leave that untouched)
            if k=1, cmd.eIsUnix=.false, length(newArr[k]=2), substr(newArr[k],2,1)=":" then
               nop
            else
            do
               newArr~remove(k)  -- remove directory
               k=k-1             -- adjust k
               iterate           -- process next chunk
            end

         end
      end
      k=k+1                         -- increment index
      newArr[k]=w                   -- save
   end

   return newArr~makeString("LINE",j.file.separator)




/*  2008-05-17, ---rgf
    returns .true, if path to Rexx is installed in HKCU (user settings), .false else;
    using a routine to deal with a raised syntax condition (if key is not available in registry)
*/
::routine isRexxInstalledInHKCU
  signal on syntax               -- gets triggered if key does not exist in registry
  val=.OleObject~new("WScript.Shell")~regRead("HKCU\Environment\Path")
  paths=val~makeArray(";")       -- create array from string, use path delimiter (";")

  do p over paths                -- iterate over individual paths
      -- if "rexx.exe" is contained in a directory, then return .true
     if stream(p~strip"\rexx.exe", "C", "QUERY EXISTS")<>"" then
        return .true
  end
  return .false                  -- not found

syntax:
  return .false            -- HKCU-PATH does not exist at all !




/*
---- for testing canonizePath():
l=.list~of( "c:\eins\zwei\drei\vier", -
            "c:\eins\zwei\drei\.."  , -
            "c:\eins\zwei\..\.."    , -
            "c:\eins\zwei\..\..\a"  , -
            "c:\eins\..\..\.."      , -
            "c:\..\..\..\.."        , -
            "--------------"        , -
              "\eins\zwei\drei\.."  , -
              "\eins\zwei\..\.."    , -
              "\eins\zwei\..\..\a"  , -
              "\eins\..\..\.."      , -
              "\..\..\..\.."        , -
            "--------------"        , -
               "eins\zwei\drei\.."  , -
               "eins\zwei\..\.."    , -
               "eins\zwei\..\..\a"  , -
               "eins\..\..\.."      , -
               "..\..\..\.."        , -
            "--------------"        , -
            "c:\eins\zwei\drei\."  , -
            "c:\eins\zwei\.\."    , -
            "c:\eins\zwei\.\.\a"    , -
            "c:\eins\.\.\."      , -
            "c:\.\.\.\."        , -
            "--------------"        , -
              "\eins\zwei\drei\."  , -
              "\eins\zwei\.\."    , -
              "\eins\zwei\.\.\a"    , -
              "\eins\.\.\."      , -
              "\.\.\.\."        , -
            "--------------"        , -
               "eins\zwei\drei\."  , -
               "eins\zwei\.\.\a"  , -
               "eins\zwei\.\."    , -
               "eins\.\.\."      , -
               ".\.\.\."              )

do it over l
   say (it)~left(22) "->" canonizePath(it)
end
exit
----
*/


/*

on UNIX the extension for dynamic link libraries can be:

.so      ibm-aix solaris sparc linux freebsd netbsd openbsd dec-osf nto-qnx qnx beos unknown
.dylib   apple-darwin
.sl      hp-hpux

on Windows:

.dll     windows cygnus


looking for Unix:    libjvm.so | libjvm.dylib | libjvm.sl
            Windows: jvm.dll

*/


