package org.rexxla.bsf.engines.rexx;

import java.util.*;
import java.io.*;
import java.lang.reflect.*;
import com.ibm.bsf.*;
import com.ibm.bsf.util.*;
import com.ibm.bsf.util.type.*;


/**
  * This class manages the calls from Rexx or Object Rexx into Java.
  * (Any scripting language being able to pass Strings to and
  * from Java will be able to use this interface, making it <em>really</em> easy to interface
  * with Java.)
 *
  * Proof of concept by Peter Kalender (Oct 2000 to Feb 2001, student of Rony G. Flatscher
 *  at the University of Essen).
  *
  *
 * <pre>------------------------ Apache Version 2.0 license -------------------------
 *    Copyright (C) 2001-2006 Rony G. Flatscher
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        <a href="http://www.apache.org/licenses/LICENSE-2.0">http://www.apache.org/licenses/LICENSE-2.0</a>
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * ----------------------------------------------------------------------------- </pre>
 *
  * @version 2.6, 2006-01-01
  * @author Rony G. Flatscher (<a href="http://www.wu-wien.ac.at">WU-Wien/Wirtschaftsuniversit&auml;t Wien</a>, <a href="http://www.wu-wien.ac.at/english">http://www.wu-wien.ac.at/english</a>)
  */

 /*
     2005-11-28, ---rgf, changed license to Apache 2.0
     2005-11-06, ---rgf, added a TypeConvertor to indicate that converstion from "java.lang.String"
                         "java.lang.Object" is o.k. for Rexx (otherwise an error occurs)
     2005-07-30, ---rgf, added method pp() - "pretty print for debug-output"
     2005-07-07, ---rgf, fixed bug in coercing boolean/Boolean values
     2005-06-18, ---rgf, fixed bug if interface is sought and type is String
     2005-06-10, ---rgf, sending attribute message to interface class now works
     2005-05-06, ---rgf, added GET_FIELD_STRICT
 */


class RexxAndJava {
    /** Version string indicating version of this class (majorVersion*100+minorVersion
     *  concatenated with a dot and the sorted date of last change.
     */
    static public String version = "206.20060101";

    /** Defines the version string. */
    static final private String RAJ_VERSION = "256.20060101 org.rexxla.bsf.engines.rexx (com.ibm.bsf)";

    /** Defines the name of DLL to be used to resolve the native functions. */
    static final protected String DLL_NAME = "BSF4Rexx";

    /** Defines the string to indicate an 'eventText' is passed in: &quot;&lt;eventText&gt;&quot;. */
    static final protected String EVENT_TEXT = "<eventText>";



    /** Defines the string which represents <code>null</code>, by default
        <code>&quot;.NIL&quot;</code>. Arguments given by Rexx can in addition
        be indicated to be <code>null</code> by merely omitting them on the Rexx side.
      */
    static protected String null4Rexx = ".NIL";


    /** Allows Synchronized access to the vector storing the
      * text sent by the eventListeners.
      */
    protected SyncPutGet eventTextList=new SyncPutGet() ;


    // ---rgf, 2003-05-09
    /** <code>true</code> if setting accessibility rights is available, <code>false</code> else.
     */
    static boolean bMethodHasSetAccessible=false;

    /** Stores the BSF manager. */
    BSFManager mgr;

    /** Stores the Rexx engine. */
    BSFEngine  RexxEngine;

    /** Stores the eventTextVector. */

    /** Stores the number of references from Object Rexx to Java objects using
      * <code>beanName</code> as the key. This way it becomes possible for the
      * Object Rexx implementation using <code>&quot;BSF.cls&quot;</code> to
      * transparently create Object Rexx proxy objects, if objects are returned
      * from Java method invocations and to transparently unregister them, when
      * the Object Rexx proxy objects are garbage collected.
      */
    Hashtable orxReferences = new Hashtable();


    /** Allows storing successfully reflected methods. If a method is invoked
     *  on a bean, it is being retrieved via reflection, a relatively slow process.
     *  This Hashtable allows re-usage of method objects for invocation purposes.
     *  The format is: "className"+" "+"methodname"+" "+nr_of_method_args, the
     *  stored object is a vector of arrays of class-objects representing the
     *  needed argument types. (It is possible that there are different signatures
     *  for a method, although the number of arguments is the same.)
     *  If this is too slow, one could use a different scheme.
     */
    Hashtable methodReferences = new Hashtable();


    /**
      * To store the default TypeConvertorRegistry, if needed.
      */
    TypeConvertorRegistry glob_tcr = new TypeConvertorRegistry();;  // will include default set of type convertors


    /** Load the DLL, which allows interfacing Java with Rexx or Object Rexx, determines
     *  whether setting accessibility rights is supported.
     */
    static
    {
        System.loadLibrary( DLL_NAME );

        // ---rgf, 2003-05-09, determine whether changing accessibility of Methods is possible
        Class mc=Method.class;            // get the "Method" class object
        Class arg[]={boolean.class};      // define an array with the primitive "boolean" pseudo class object
        try {
            Object o=mc.getMethod("setAccessible", arg ); // is this method available?
            bMethodHasSetAccessible=true; // no exception, hence method exists
        }
        catch (Exception e)
        {
            bMethodHasSetAccessible=false;// exception occurred, hence method does not exist
        }


    }




    /** Constructor to receive and store the BSF manager and the Rexx engine.
     * <p>
     *
     * Preregisters the most important class objects. Referring to them is possible
     *  by using the following strings from Rexx (case *is* significant!):
     *
     *  <ul>
     *      <li> &quot;Class.class&quot;,
     *      <li> &quot;Object.class&quot;,
     *      <li> &quot;Method.class&quot;,
     *
     *      <li> &quot;Array.class&quot;,
     *      <li> &quot;String.class&quot;,
     *      <li> &quot;System.class&quot;,
     *
     *      <li> &quot;Boolean.class&quot;,
     *      <li> &quot;boolean.class&quot;,
     *      <li> &quot;Byte.class&quot;,
     *      <li> &quot;byte.class&quot;,
     *      <li> &quot;Character.class&quot;,
     *      <li> &quot;char.class&quot;,
     *      <li> &quot;Double.class&quot;,
     *      <li> &quot;double.class&quot;,
     *      <li> &quot;Integer.class&quot;,
     *      <li> &quot;int.class&quot;,
     *      <li> &quot;Long.class&quot;,
     *      <li> &quot;long.class&quot;,
     *      <li> &quot;Float.class&quot;,
     *      <li> &quot;float.class&quot;,
     *      <li> &quot;Short.class&quot;,
     *      <li> &quot;short.class&quot;,
     *      <li> &quot;Void.class&quot;,
     *      <li> &quot;void.class&quot;.
     *  </ul>
     */
    RexxAndJava (BSFManager mgr, BSFEngine rengine)
    {
        this.mgr        = mgr;     //the current BSFManager
        this.RexxEngine = rengine; //the current RexxEngine

        // 2003-01-26, ---rgf, pre-register important class objects for easier reference from Rexx
        mgr.registerBean("Class.class",     Class.class);
        mgr.registerBean("Object.class",    Object.class);
        mgr.registerBean("Method.class",    Method.class);

        mgr.registerBean("Array.class",     Array.class);
        mgr.registerBean("String.class",    String.class);
        mgr.registerBean("System.class",    System.class);

        mgr.registerBean("Boolean.class",   Boolean.class);
        mgr.registerBean("boolean.class",   boolean.class);
        mgr.registerBean("Byte.class",      Byte.class);
        mgr.registerBean("byte.class",      byte.class);
        mgr.registerBean("char.class",      char.class);
        mgr.registerBean("Character.class", Character.class);
        mgr.registerBean("Double.class",    Double.class);
        mgr.registerBean("double.class",    double.class);
        mgr.registerBean("Float.class",     Float.class);
        mgr.registerBean("float.class",     float.class);
        mgr.registerBean("int.class",       int.class);
        mgr.registerBean("Integer.class",   Integer.class);
        mgr.registerBean("Long.class",      Long.class);
        mgr.registerBean("long.class",      long.class);
        mgr.registerBean("Short.class",     Short.class);
        mgr.registerBean("short.class",     short.class);
        mgr.registerBean("Void.class",      Void.class);
        mgr.registerBean("void.class",      void.class);


        // for conversions of java.lang.String -> java.lang.Object, RexxAndJava.java needs an explicit
        // type convertor; rgf, 2005-11-06
        {
                // register conversion of java.lang.String->java.lang.Object here, no need seen
                // to enclose it into the standard type convertors in class TypeConvertorRegistry.java
             TypeConvertor tc = new TypeConvertor () {
              public Object convert (Class from, Class to, Object obj) {
                  return obj;
              }

              public String getCodeGenString() {
                    return "(Class from, Class to, Object obj) {\n" +
                               "return obj;\n" +
                               "}";
              }
            };
                 // register this type-convertor
            glob_tcr.register (String.class, Object.class, tc); // rgf, 2005-07-30
        }
    }


    /** This native method makes sure that the function 'BSF' is
      * registered with Rexx, making it thereby available to all
      * Rexx and Object Rexx programs, called from Java.
      */
    public native int jniRegisterBSF ();        // changed to int (---rgf, 2001-05-25)



    /** This native method Calls the Rexx interpreter,
      * having it execute the passed in
      * <em>script</em> with the given array of arguments.
      *
      * @param script a String consisting of Rexx statements separated by
      *               by line-end characters (';' or CR-LF).
      *
      * @param args an array of Strings representing the arguments
      *               or <code>null</code>, if no arguments are available.
      *
      * @return a String containing the result or <code>null</code>, if
      *         no result is returned or the Rexx interpreter stopped
      *         execution due to a runtime error.
      */
    public native String jniRexxStart (String script, String[] args);



    /** @deprecated
      * This native method waits for the Object Rexx interpreter to terminate and waits until
      * all of the cleanup (e.g. of Object Rexx) is done.
      *
      * @return returns always 0. (Just there to allow future return codes, if necessary.)
      */
    public native int jniRexxWaitForTermination ( );


    /** @deprecated
      * This native method indicates whether the Object Rexx interpreter is still running or not.
      *
      * @return returns always 0. (Just there to allow future return codes, if necessary.)
      */
    public native int jniRexxDidRexxTerminate ( );


    /**  rgf, 2003-01-15, used for allowing the BSF4Rexx-DLL to get at *env, *obj and then to mid etc.
     * */
    public native int jniInitialize4Rexx() ;

        // define the constants for the desired function
private static final int           ADD_EVENT_LISTENER        =  1  ;
private static final int           ARRAY_AT                  =  2  ;
private static final int           ARRAY_LENGTH              =  3  ;
private static final int           ARRAY_PUT                 =  4  ;
private static final int           ARRAY_PUT_STRICT          =  5  ;
private static final int           CREATE_ARRAY_OBJECT       =  6  ;
private static final int           EXIT                      =  7  ;
private static final int           GET_FIELD_VALUE           =  8  ;
private static final int           GET_FIELD_VALUE_STRICT    =  9  ;    // 2005-05-06, ---rgf
private static final int           GET_PROPERTY_VALUE        =  10 ;
private static final int           GET_STATIC_VALUE          =  11 ;
private static final int           GET_STATIC_VALUE_STRICT   =  12 ;
private static final int           GET_VERSION               =  13 ;
private static final int           INVOKE                    =  14 ;
private static final int           INVOKE_STRICT             =  15 ;
private static final int           LOOKUP_BEAN               =  16 ;
private static final int           POLL_EVENT_TEXT           =  17 ;
private static final int           POST_EVENT_TEXT           =  18 ;
private static final int           REGISTER_BEAN             =  19 ;
private static final int           NEW                       =  19 ;    // 2005-06-17, ---rgf: synonym for REGISTER_BEAN, hence same value
private static final int           REGISTER_BEAN_STRICT      =  20 ;
private static final int           NEW_STRICT                =  20 ;    // 2005-06-17, ---rgf: synonym for REGISTER_BEAN_STRICT, hence same value
private static final int           SET_FIELD_VALUE           =  21 ;
private static final int           SET_FIELD_VALUE_STRICT    =  22 ;
private static final int           SET_PROPERTY_VALUE        =  23 ;
private static final int           SET_PROPERTY_VALUE_STRICT =  24 ;
private static final int           SET_REXX_NULL_STRING      =  25 ;
private static final int           SLEEP                     =  26 ;
private static final int           UNREGISTER_BEAN           =  27 ;
private static final int           WRAP_ARRAY_OBJECT         =  28 ;
private static final int           WRAP_ENUMERATION_OBJECT   =  29 ;

        // define the constant strings for the function names passed from Rexx
private static final String STRING_ADD_EVENT_LISTENER        = "addEventListener"        ;
private static final String STRING_ARRAY_AT                  = "arrayAt"                 ;
private static final String STRING_ARRAY_LENGTH              = "arrayLength"             ;
private static final String STRING_ARRAY_PUT                 = "arrayPut"                ;
private static final String STRING_ARRAY_PUT_STRICT          = "arrayPutStrict"          ;
private static final String STRING_CREATE_ARRAY_OBJECT       = "createArray"             ; // rgf, 2003-02-17
private static final String STRING_EXIT                      = "exit"                    ;
private static final String STRING_GET_FIELD_VALUE           = "getFieldValue"           ;
private static final String STRING_GET_FIELD_VALUE_STRICT    = "getFieldValueStrict"     ; // rgf, 2005-05-06
private static final String STRING_GET_PROPERTY_VALUE        = "getPropertyValue"        ;
private static final String STRING_GET_STATIC_VALUE          = "getStaticValue"          ;
private static final String STRING_GET_STATIC_VALUE_STRICT   = "getStaticValueStrict"    ; // rgf, 2005-05-06
private static final String STRING_GET_VERSION               = "version"                 ; // rgf, 2003-03-10
private static final String STRING_INVOKE                    = "invoke"                  ;
private static final String STRING_INVOKE_STRICT             = "invokeStrict"            ; // rgf, 2003-05-07
private static final String STRING_LOOKUP_BEAN               = "lookupBean"              ;
private static final String STRING_POLL_EVENT_TEXT           = "pollEventText"           ;
private static final String STRING_POST_EVENT_TEXT           = "postEventText"           ;
private static final String STRING_REGISTER_BEAN             = "registerBean"            ;
private static final String STRING_NEW                       = "new"                     ; // rgf, 2005-06-17
private static final String STRING_REGISTER_BEAN_STRICT      = "registerBeanStrict"      ; // rgf, 2003-05-08
private static final String STRING_NEW_STRICT                = "newStrict"               ; // rgf, 2005-06-17
private static final String STRING_SET_FIELD_VALUE           = "setFieldValue"           ;
private static final String STRING_SET_FIELD_VALUE_STRICT    = "setFieldValueStrict"     ; // rgf, 2003-05-10
private static final String STRING_SET_PROPERTY_VALUE        = "setPropertyValue"        ;
private static final String STRING_SET_PROPERTY_VALUE_STRICT = "setPropertyValueStrict"  ; // rgf, 2003-05-10
private static final String STRING_SET_REXX_NULL_STRING      = "setRexxNullString"       ;
private static final String STRING_SLEEP                     = "sleep"                   ;
private static final String STRING_UNREGISTER_BEAN           = "unregisterBean"          ;
private static final String STRING_WRAP_ARRAY_OBJECT         = "wrapArray"               ; // rgf, 2003-02-09
private static final String STRING_WRAP_ENUMERATION_OBJECT   = "wrapEnumeration"         ; // rgf, 2003-02-12


    /** Entry point for Rexx calls into Java. This method carries out
      * all the necessary type conversions and finally calls/invokes the
      * indicated Java method.
     *
      * <p>
     *
     * <hr width="50%">
     *
      <em>&quot;typeIndicator&quot;</em>: If a call needs to explicitly indicate the type of Rexx arguments meant
      for invoking a Java
      * method (indicated by a trailing "Strict" in the subfunction name),
      * than the following strings/characters must be used for defining
      * the type of the argument immediately preceding the respective
        argument (minimum letters needed are uppercased and printed in bold):
      *
      *<p>

        <table border="1" cellpadding="3" cellspacing="0" align="center">
        <tr class="TableHeadingColor">
        <td> <strong> TypeIndicator</strong>
        <td> <strong> Following Argument is of Type:
             </strong>

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>BO</strong>olean&quot; </code>
        <td> <code> &quot;boolean&quot;</code> i.e. the value <code>0</code> (false) or <code>1</code> (true)

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>BY</strong>te&quot; </code>
        <td> <code> &quot;byte&quot;</code> a byte value

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>C</strong>har&quot; </code>
        <td> <code> &quot;char&quot;</code> i.e. a single (UTF8) character

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>D</strong>ouble&quot; </code>
        <td> <code> &quot;double&quot;</code> a double value

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>F</strong>loat&quot; </code>
        <td> <code> &quot;float&quot;</code> a float value

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>I</strong>nt&quot; </code>
        <td> <code> &quot;int&quot;</code> an integer value

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>L</strong>ong&quot; </code>
        <td> <code> &quot;long&quot;</code> a long value

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>O</strong>bject&quot; </code>
        <td> &quot;Object&quot; an Object which must be registered already

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>SH</strong>ort&quot; </code>
        <td> <code> &quot;short&quot;</code> a short value

        <tr class="TableRowColor"><!-- table row -->
        <td> <code> &quot;<strong>ST</strong>ring&quot; </code>
        <td> <code> &quot;String&quot;</code> a string value (UTF8)


        </table>
        <p>

      * Hence, for the "...Strict"-subfunctions the Rexx supplied arguments
        must be given
      * in pairs by Rexx: first the type argument (from the above table)
       and then the argument itself.
     * <hr width="50%">
      *
      *  <p>The call syntax from Rexx looks like, e.g.:
      *
      * <pre>
      *    call BSF "SubFunctionName", "argument1", "argument2", "", ...
      * or:
      *    a=BSF("SubFunctionName", "argument1", "argument2", ...)
      * </pre>
      *
      *
      * <p>The following BSF methods are made available to Rexx and
      * can therefore be indicated as the first argument (the 'subfunction') to the 'BSF'
      * call from Rexx:
      * <p>
      *
     *
     *
     *
     *

        <table border="1" cellpadding="3" cellspacing="0">
        <tr class="TableHeadingColor">
        <td> <strong>SubFunctionName </strong>
        <td> <strong> Argument     </strong>
        <td> <strong> Description  </strong>


        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;addEventListener&quot; </code>
        <td colspan="2">Adds an event listener to a bean and defines the <code>eventText</code>
                        to be generated. These generated texts can be retrieved with the
                        function <code>pollEventText</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of a registered bean which should get monitored.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>eventSetName </code>
        <td> Mandatory, name of the set the event belongs to. This name is used
             to create an event adapter at runtime.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>filter </code>
        <td> Mandatory, name of the actual event which should cause a notification.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>eventText </code>
        <td> Mandatory, text which should be sent by this event listener, each
             time the event occurs.



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;arrayAt&quot; </code>
        <td colspan="2">Allows accessing a Java array element.
                        <br>Returns the Java object at the indicated array element,
                            or <code>null</code>, if no object is stored there.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arrayObject </code>
        <td> Mandatory, name of a registered arrayObject (received by Java).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>idx1 </code>
        <td> Mandatory, integer indicating the position in dimension 1.
             <p>
             <em>Alternatively,</em> a sole Java int-array containing the 0-based index-values.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>idx2 </code>
        <td> Integer indicating the position in dimension 2 ...



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;arrayLength&quot; </code>
        <td colspan="2">Returns the value of the 'length' field of the <code>arrayObject</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arrayObject </code>
        <td> Mandatory, name of a registered arrayObject (received by Java).







        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;arrayPut&quot; </code>
        <td colspan="2">Allows to put a new value into a Java array element at the
                        indicated position.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arrayObject </code>
        <td> Mandatory, name of a registered arrayObject (received by Java).


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newValue </code>
        <td> The new value to be set at the indicated position in the array.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>idx1 </code>
        <td> Mandatory, integer indicating the position in dimension 1.
             <p>
             <em>Alternatively,</em> a sole Java int-array containing the 0-based index-values.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>...idxn </code>
        <td> Integer indicating the position in dimension n ...

     *
     *

        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;arrayPutStrict&quot; </code>
        <td colspan="2">Allows to put a new value into a Java array element at the
                        indicated position.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arrayObject </code>
        <td> Mandatory, name of a registered arrayObject (received by Java).


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>typeIndicator </code>
        <td> Type indicator for <code>newValue</code>

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newValue </code>
        <td> The new value to be set at the indicated position in the array.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>idx1 </code>
        <td> Mandatory, integer indicating the position in dimension 1.
             <p>
             <em>Alternatively,</em> a sole Java int-array containing the 0-based index-values.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>... idxn </code>
        <td> Integer indicating the position in dimension n ...

     *
     *


        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;createArray&quot; </code>
        <td colspan="2">Creates a Java array of the given <code>componentType</code> and number of
                       dimensions.
                        <br>Returns the created Java array object.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>componentType</code>
        <td> Mandatory, name of a registered Java class object. For convenience, you may use any
             of the pre-registered beans named (note case of first letter!):<br>

                 &quot;<code>Class.class</code>&quot;,
                 &quot;<code>Object.class</code>&quot;,
                 &quot;<code>Method.class</code>&quot;,

                 &quot;<code>Array.class</code>&quot;,
                 &quot;<code>String.class</code>&quot;,
                 &quot;<code>System.class</code>&quot;,

                 &quot;<code>Boolean.class</code>&quot;,
                 &quot;<code>boolean.class</code>&quot; (primitive data type),
                 &quot;<code>Byte.class</code>&quot;,
                 &quot;<code>byte.class</code>&quot; (primitive data type),
                 &quot;<code>Character.class</code>&quot;,
                 &quot;<code>char.class</code>&quot; (primitive data type),
                 &quot;<code>Double.class</code>&quot;,
                 &quot;<code>double.class</code>&quot; (primitive data type),
                 &quot;<code>Integer.class</code>&quot;,
                 &quot;<code>int.class</code>&quot; (primitive data type),
                 &quot;<code>Long.class</code>&quot;,
                 &quot;<code>long.class</code>&quot; (primitive data type),
                 &quot;<code>Float.class</code>&quot;,
                 &quot;<code>float.class</code>&quot; (primitive data type),
                 &quot;<code>Short.class</code>&quot;,
                 &quot;<code>short.class</code>&quot; (primitive data type).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>capacity 1</code>
        <td> Mandatory, integer indicating the number of elements in dimension 1.
             <p>
             <em>Alternatively,</em> a sole Java int-array containing the capacity per dimension.
             The first dimension is stored with index 0, the second with index 1, ...

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>capacity 2</code>
        <td> Integer indicating the number of elements in dimension 2.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> ... <code>capacity n </code>
        <td> Integer indicating the number of elements in dimension n.



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;wrapArray&quot; </code>
        <td colspan="2">Wraps the given array up in a {@link ArrayWrapper}-object, allowing
                        easy access to information about the array like dimensions, size and
                        also getting and setting values in an easy manner.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arrayObject </code>
        <td> Mandatory, name of a registered arrayObject (received by Java).




        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;exit&quot; </code>
        <td colspan="2">Terminates the Java Virtual Machine. Allows Rexx
                        to shut down the JVM (and thereby Rexx as well).
                        In order for Rexx being able to shutdown properly
                        itself, the termination of the JVM is delayed by
                        <code>time2wait</code>.

                        <br>Returns the string
                        <code>&quot;SHUTDOWN,&nbsp;REXX&nbsp;!&quot;</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>retVal</code>
        <td> Optional integer return value. If omitted, then &quot;0&quot;
             will be used as argument for the <code>System.exit(n)</code> call.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>time2wait</code>
        <td> Optional number of milliseconds (msecs) to wait until the JVM will
             be terminated. This allows Rexx to shutdown (clean up) its side as
             well. Default value, if omitted: <code>100</code>, i.e. 100 msecs,
             i.e. 1/10th second.



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;getFieldValue&quot; </code>
        <td colspan="2">Looks up a <em>public</em> field of the registered
                        object. The case of the characters of the field name does not matter.
                        <br>
                        <em>Hint:</em> If the field cannot be found, then this subfunction assumes that
                        a getter method (field name prepended by &quot;get&quot;) exists for the field and tries to invoke one. If
                        you do not want this behaviour use subfunciton
                        <code>&quot;getFieldValueStrict&quot;</code> instead.


                        <br>Returns the value of the field.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>fieldName </code>
        <td> Mandatory, name of the <em>public</em> field of the object.



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;getFieldValueStrict&quot; </code>
        <td colspan="2">Looks up a <em>public</em> field of the registered
                        object. The case of the characters of the field name must match exactly.
                        <br>Returns the value of the field.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>fieldName </code>
        <td> Mandatory, <em>case sensitive</em> name of the <em>public</em> field of the object.






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;getPropertyValue&quot; </code> <strong>(untested!)</strong>
        <td colspan="2">Looks up a <em>property</em> of the registered
                        <em>Java Bean</em> object.
                        <br>Returns the value of the property.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the Java Bean object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>propertyName </code>
        <td> Mandatory, name of the Java Bean <em>property</em>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>index </code>
        <td> Mandatory, <code>null</code>, if Java Bean property is not indexed,
             a valid integer number else.






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;getStaticValue&quot; </code>
        <td colspan="2">Looks up a <em>public static</em> field of the
                        given class or interface. This function does <em>not instantiate</em>
                        the class (would not be possible for abstract classes
                        or interfaces). Case of <code>fieldName</code> is <em>not</em> relevant!
                        <br>Returns the value of the public static field.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>className </code>
        <td> Mandatory, name of the class, abstract class or interface to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>fieldName </code>
        <td> Mandatory, name of the <em>public static</em> field of the object.


        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;getStaticValueStrict&quot; </code>
        <td colspan="2">Looks up a <em>public static</em> field of the
                        given class or interface. This function does <em>not instantiate</em>
                        the class (would not be possible for abstract classes
                        or interfaces). Case of <code>fieldName</code> is relevant!
                        <br>Returns the value of the public static field.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>className </code>
        <td> Mandatory, name of the class, abstract class or interface to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>fieldName </code>
        <td> Mandatory, <em>case sensitive</em> name of the <em>public static</em> field of the object.





        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;invoke&quot; </code>
        <td colspan="2">Invokes (calls) the method with the given arguments
                        on the bean.
                        <br>

                        <em>Hint:</em> if the method could not be found and has no arguments,
                        this subfunction assumes that the value of a field is sought. Therefore,
                        it employs the subfunction <code>&quot;getFieldValue&quot;</code> using
                        the <code>methodName</code> as the <code>fieldName</code>.

                        If you do not want to use this
                        feature, then employ the subfunction <code>&quot;invokeStrict&quot;</code> instead.

                        <br>Returns the result of the method invocation or
                            <code>null</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of a registered bean on which the method (function)
             should get invoked.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>methodName </code>
        <td> Mandatory, name of the method which should get invoked. <code>methodName</code> is <em>not</em>
             case sensitive!

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arg # 1</code>
        <td> Argument # 1 (if given).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>... arg # n</code>
        <td> Argument # n (if given).






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;invokeStrict&quot; </code>
        <td colspan="2">Invokes (calls) the method with the given arguments
                        on the bean.
                        The case of the characters of the method name must match exactly.
                        Each Rexx argument <em>must</em> be preceeded by its Java type.

                        <br>Returns the result of the method invocation or
                            <code>null</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of a registered bean on which the method (function)
             should get invoked.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>methodName </code>
        <td> Mandatory, <em>case sensitive</em> name of the method which should get invoked.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>typeIndicator # 1</code>
        <td> Type indicator for argument # 1 (if given).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arg # 1</code>
        <td> The argument # 1 itself (if given).

        <tr class="TableRowColor"><!-- table row -->
        <td >
        <td colspan="2"> <em>Any additional arguments must be given in pairs of
                       &quot;typeIndicator&quot; and &quot;arg&quot; as well.</em>

     *
     *
     *

        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;lookupBean&quot; </code>
        <td colspan="2">Looks up an object in the BSF
            manager object registry. (Allows for retrieving pre-<em>declared
            beans</em> as well.)
            <br>Returns the object or <code>null</code>, if not found.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the object to be looked up. Returns
             the object rendered to a string or <code>null</code>, if not found.






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;pollEventText&quot; </code>
        <td colspan="2">Polls for the eventText defined in the above subfunction
                        &quot;<code>addEventListener</code>&quot; placed
                        into a queue by the appropriate event listeners,
                        or posted via &quot;<code>postEventText</code>&quot; below.
                        <br>

                        If no timeout value is supplied, &quot;0&quot;
                        will be used by default, which blocks the return of this function
                        (forever) until some text is available for return.

                        <br>Returns available <code>eventText</code>, or if a
                            timeout occurred while waiting <code>null</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>timeOut </code>
        <td> Optional, timeout in milliseconds, i.e. time to wait for
             an <code>eventText</code> to appear, before returning.






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;postEventText&quot; </code>
        <td colspan="2">Adds <code>eventText</code> to the <code>eventText</code> vector (list),
                        can be retrieved via <code>pollEventText</code>.
                       <br>Returns the received <code>eventText</code>

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>eventText </code>
        <td> Mandatory, text which should be put into the <code>eventText</code> vector.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>priority </code>
        <td> Optional, priority level of <code>eventText</code>, can be one of:
          *        <p><ul>
                   <li>&quot;<code>2</code>&quot;: high ('alarm level') priority.
                   <li>&quot;<code>1</code>&quot;: normal (<strong>default</strong>) priority, and
                   <li>&quot;<code>0</code>&quot;: low ('batch level') priority,
                   </ul>



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;registerBean&quot; </code> or<br>
             <code>&quot;new&quot;</code> (a synonym)
        <td colspan="2">Creates a new object (instantiating the given Java class
                        with the given arguments) and registers it using the
                        given <em>beanName</em>.
                        <br>Returns <em>beanName</em>
                        (always use this returned beanName for looking up the
                        object in the registry).


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Name of the object to be created and registered.
             if no name is given (empty string or <code>null</code>)
             then a unique one will be created and returned;
             the code to create the unique name is the same
             as the documented &quot;Object.hashCode()&quot;:
             <br><code>
             o.getClass().getName() + &quot;@&quot;
             + Integer.toHexString(o.hashCode())</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanType</code>
        <td> Mandatory, type (Java class name) of the bean (Java object) to be
             created and registered.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arg # 1</code>
        <td> The argument # 1 (if given).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>... arg # n</code>
        <td> The argument # n (if given).





        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;registerBeanStrict&quot; </code>  or<br>
             <code>&quot;newStrict&quot;</code> (a synonym)
        <td colspan="2">Creates a new object (instantiating the given Java class
                        with the given arguments) and registers it using the
                        given <em>beanName</em>.
                        <br>Returns <em>beanName</em>
                        (always use this returned beanName for looking up the
                        object in the registry).


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Name of the object to be created and registered.
             if no name is given (empty string or <code>null</code>)
             then a unique one will be created and returned;
             the code to create the unique name is the same
             as the documented &quot;Object.hashCode()&quot;:
             <br><code>
             o.getClass().getName() + &quot;@&quot;
             + Integer.toHexString(o.hashCode())</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanType</code>
        <td> Mandatory, type (Java class name) of the bean (Java object) to be
             created and registered.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>typeIndicator # 1</code>
        <td> Type indicator for argument # 1 (if given).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>arg # 1</code>
        <td> The argument # 1 (if given).

        <tr class="TableRowColor"><!-- table row -->
        <td >
        <td colspan="2"> <em>Any additional arguments must be given in pairs of
                       &quot;typeIndicator&quot; and &quot;arg&quot; as well.</em>





        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;setFieldValue&quot; </code>
        <td colspan="2">Looks up a <em>public</em> field of the registered
                        object and sets it to the argument's value.
                        <br>

                        <em>Hint:</em> if the field could not be found, this subfunction
                        automatically attempts to employ a setter method (<code>fieldName</code>
                        prepended with the string &quot;set&quot;). If you do not want to use this
                        feature, then employ the subfunction <code>&quot;setFieldValueStrict&quot;</code> instead.

                        <br>Returns the new value of the field.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>fieldName </code>
        <td> Mandatory, name of the <em>public</em> field of the object.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newValue </code>
        <td> The new value the field gets set to.




        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;setFieldValueStrict&quot; </code>
        <td colspan="2">Looks up a <em>public</em> field of the registered
                        object and sets it to the argument's value.
                        <br>Returns the new value of the field.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>fieldName </code>
        <td> Mandatory, <em>case sensitive</em> name of the <em>public</em> field of the object.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>typeIndicator </code>
        <td> <em>optional</em> type indicator for the immediately following argument
             (<code>newValue</code>) or <code>newValue</code>.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newValue </code>
        <td> The new value the field gets set to. (If this argument is given, then the
             previous argument is interpreted as a <code>typeIndicator </code>.)






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;setPropertyValue&quot; </code> <strong>(untested!)</strong>
        <td colspan="2">Looks up a <em>property</em> of the registered
                        <em>Java Bean</em> object and sets it to the argument's value.
                        <br>Returns the new value of the property.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the Java Bean object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>propertyName </code>
        <td> Mandatory, name of the Java Bean <em>property</em>.


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>index </code>
        <td> Mandatory, <code>null</code>, if Java Bean property is not indexed,
             a valid integer number else.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newValue </code>
        <td> The new value the property gets set to.






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;setPropertyValueStrict&quot; </code> <strong>(untested!)</strong>
        <td colspan="2">Looks up a <em>property</em> of the registered
                        <em>Java Bean</em> object and sets it to the argument's value.
                        <br>Returns the new value of the property.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of the Java Bean object to be looked up.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>propertyName </code>
        <td> Mandatory, name of the Java Bean <em>property</em>.


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>index </code>
        <td> Mandatory, <code>null</code>, if Java Bean property is not indexed,
             a valid integer number else.

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>typeIndicator </code>
        <td> Type indicator for the immediately following argument
             (<code>newValue</code>).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newValue </code>
        <td> The new value the property gets set to.






        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;setRexxNullString&quot;</code>
        <td colspan="2">Sets the field {@link #null4Rexx} to the string which serves
                        as the representation for <code>null</code>.
                        <ul>
                        <li>The Java side will return this value to Rexx to indicate a Java
                            <code>null</code> (&quot;no value&quot;). This conversion is
                            done in method {@link #makeString4Rexx(Object o)}.

                        <li>The Rexx side can use this string for
                            an argument for which no value is supplied.
                            This option is available in addition to the standard Rexx option
                            of just leaving the missing argument empty. The conversion of
                            Rexx arguments to Java is done in
                            {@link #convFromRexx(String type, String value)}.
                        </ul>


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>newString </code>
        <td> Mandatory, the new string to be used to indicate a Java
            <code>null</code>.





        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;sleep&quot;</code>

        <td colspan="2">Sleeps <code>time2sleep</code> seconds before returning.


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code> time2sleep </code>
        <td align="left"> Mandatory, indicates seconds to sleep before returning. <code>time2sleep</code>.
                        The sleeping occurs with an attempted resolution of milliseconds,
                        being calculated like:
                        <code> (long) ((double)time2sleep*1000+0.5d)</code>, e.g. <br>
                   <ul>
                   <li><code>3</code> (three seconds, 3000 msec),
                   <li><code>600</code> (600 seconds, i.e. 10 minutes, 600000 msec),
                   <li><code>0.5</code> (5/10 second, i.e. half a second, 500 msec),
                   <li><code>0.01</code> (1/100 second, 10 msec),
                   <li><code>0.001</code> (1/1000 second,  1 msec),
                   <li><code>0.0005</code> (rounded up, ie. 1/1000 second, 1 msec),
                   <li><code>0.0004</code> (rounded down, ie. 0/1000 second, 0 msec).
                   </ul>

                   <p><em>Hint:</em> If available, use Rexx' <code>SysSleep()</code> function
                   instead (it is usually more exact).




        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;unregisterBean&quot; </code>
        <td colspan="2">Unregisters an object from the BSF manager's
                        object registry.
                        <br>Returns the former beanName of the just unregistered object.


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>beanName </code>
        <td> Mandatory, name of bean (Java object) to be unregistered.



        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code>&quot;version&quot; </code>
        <td colspan="2">Returns the version of this program in the form of
              &quot;<code>version.YYYYMMMDDD package-name-of-engine info</code>&quot;, e.g.

              &quot;<code>254.20060101 org.rexxla.bsf.engines.rexx (com.ibm.bsf)</code>&quot;
              for IBM's BSF or
              &quot;<code>254.20060101 org.rexxla.bsf.engines.rexx (com.ibm.bsf)</code>&quot;
              for Apache's BSF.


        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code></code>
        <td> There exists no argument for this subfunction.




        <!-- -------------------------------------------------------- -->
        <tr class="TableHeadingColor"><!-- table row -->
        <td> <code> &quot;wrapEnumeration&quot; </code>
        <td colspan="2">Wraps the given object up in a {@link EnumerationWrapper}-object, allowing
                        Java 1.1 programs to use the methods of the Enumeration interface
                        (the problem: Java 1.1 raises an IllegalAccessException, if accessing
                        methods of inner classes; starting with 1.2 one can come by this bug
                        under program control).

        <tr class="TableRowColor"><!-- table row -->
        <td>
        <td align="right"> <code>enumerationObject </code>
        <td> Mandatory, name of a registered enumerationObject (received by Java).





        </table>
        <p>
      *
      *  @return A Rexx string or <code>null</code>.
      *
      */

    public String javaCallBSF (String args[])  throws BSFException
    {
/* System.err.println("\n---> RexxAndJava.javaCallBSF(), args[0]=["+args[0]+"]."); */

       // determine which function to invoke

       boolean bStrongTyping=false; // set to true, if Java types preceed the individual Rexx arguments
       int choice = 0;

            if (args[0].equalsIgnoreCase( STRING_INVOKE                    )) choice = INVOKE                    ;
       else if (args[0].equalsIgnoreCase( STRING_INVOKE_STRICT             )) choice = INVOKE_STRICT             ;

       else if (args[0].equalsIgnoreCase( STRING_REGISTER_BEAN             )) choice = REGISTER_BEAN             ;
       else if (args[0].equalsIgnoreCase( STRING_NEW                       )) choice = NEW                       ;
       else if (args[0].equalsIgnoreCase( STRING_REGISTER_BEAN_STRICT      )) choice = REGISTER_BEAN_STRICT      ;
       else if (args[0].equalsIgnoreCase( STRING_NEW_STRICT                )) choice = NEW_STRICT                ;

       else if (args[0].equalsIgnoreCase( STRING_LOOKUP_BEAN               )) choice = LOOKUP_BEAN               ;
       else if (args[0].equalsIgnoreCase( STRING_UNREGISTER_BEAN           )) choice = UNREGISTER_BEAN           ;

       else if (args[0].equalsIgnoreCase( STRING_ADD_EVENT_LISTENER        )) choice = ADD_EVENT_LISTENER        ;
       else if (args[0].equalsIgnoreCase( STRING_POLL_EVENT_TEXT           )) choice = POLL_EVENT_TEXT           ;
       else if (args[0].equalsIgnoreCase( STRING_POST_EVENT_TEXT           )) choice = POST_EVENT_TEXT           ;

       else if (args[0].equalsIgnoreCase( STRING_GET_FIELD_VALUE           )) choice = GET_FIELD_VALUE           ;
       else if (args[0].equalsIgnoreCase( STRING_GET_FIELD_VALUE_STRICT    )) choice = GET_FIELD_VALUE_STRICT    ; // rgf, 2005-05-06
       else if (args[0].equalsIgnoreCase( STRING_GET_STATIC_VALUE          )) choice = GET_STATIC_VALUE          ;
       else if (args[0].equalsIgnoreCase( STRING_GET_STATIC_VALUE_STRICT   )) choice = GET_STATIC_VALUE_STRICT   ;

       else if (args[0].equalsIgnoreCase( STRING_EXIT                      )) choice = EXIT                      ;

       else if (args[0].equalsIgnoreCase( STRING_SET_FIELD_VALUE           )) choice = SET_FIELD_VALUE           ;
       else if (args[0].equalsIgnoreCase( STRING_SET_FIELD_VALUE_STRICT    )) choice = SET_FIELD_VALUE_STRICT    ;

       else if (args[0].equalsIgnoreCase( STRING_ARRAY_AT                  )) choice = ARRAY_AT                  ;
       else if (args[0].equalsIgnoreCase( STRING_ARRAY_LENGTH              )) choice = ARRAY_LENGTH              ;
       else if (args[0].equalsIgnoreCase( STRING_ARRAY_PUT                 )) choice = ARRAY_PUT                 ;
       else if (args[0].equalsIgnoreCase( STRING_ARRAY_PUT_STRICT          )) choice = ARRAY_PUT_STRICT          ;
       else if (args[0].equalsIgnoreCase( STRING_CREATE_ARRAY_OBJECT       )) choice = CREATE_ARRAY_OBJECT       ;

       else if (args[0].equalsIgnoreCase( STRING_GET_PROPERTY_VALUE        )) choice = GET_PROPERTY_VALUE        ;
       else if (args[0].equalsIgnoreCase( STRING_SET_PROPERTY_VALUE        )) choice = SET_PROPERTY_VALUE_STRICT ;
       else if (args[0].equalsIgnoreCase( STRING_SET_PROPERTY_VALUE_STRICT )) choice = SET_PROPERTY_VALUE_STRICT ;

       else if (args[0].equalsIgnoreCase( STRING_SET_REXX_NULL_STRING      )) choice = SET_REXX_NULL_STRING      ;
       else if (args[0].equalsIgnoreCase( STRING_SLEEP                     )) choice = SLEEP                     ;

       else if (args[0].equalsIgnoreCase( STRING_WRAP_ARRAY_OBJECT         )) choice = WRAP_ARRAY_OBJECT         ;
       else if (args[0].equalsIgnoreCase( STRING_WRAP_ENUMERATION_OBJECT   )) choice = WRAP_ENUMERATION_OBJECT   ;

       else if (args[0].equalsIgnoreCase( STRING_GET_VERSION               )) choice = GET_VERSION               ;


        // ==================================================================
        // ==================================================================

       switch (choice)
       {
          case ADD_EVENT_LISTENER :
                  {
                  /* ********************************************************
                   * call BSF 'addEventListener', beanName, eventSetName, filter, script
                   *
                   * - beanName: the bean that fires the event
                   * - eventSetName: the set the event belongs to
                   * - filter: the actual event
                   * - script: the script to be run, if the event fires
                   *********************************************************/
                      if(args.length!=5)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+" invalid # of args; usage: call BSF " +
                              "'"+args[0]+"', beanName, eventSetName, filter, script)");
                      }

                      Object bean = mgr.lookupBean(args[1]);

                      if(bean==null)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": object '"+args[1]+"' not registered");
                      }

                      args[3] = args[3].equals("") ? null : args[3];
                      try
                      {
                          EngineUtils.addEventListener(bean, args[2], args[3],
                                                       RexxEngine, mgr, EVENT_TEXT,
                                                       0, 0, args[4]);
                          return "1";   // return .true
                      }
                      catch (BSFException e)
                      {
//                          e.printStackTrace ();
                          System.err.println('"'+args[0]+'"'+": could not add 'eventText', got BSF exception: " + e.getMessage ());
                      }
                  }
        // ==================================================================

       case CREATE_ARRAY_OBJECT:
           /*******^**************************************************
            * value = BSF( 'createArray', componentType, idx1 [, idx2 [, idx3 [, idx4 [, idx5]...]]] )
            *
            * - componentType: class object
            * - idx1 ... idx5: maximum entries per indicated dimension (idx1...idx5...)
            *   alternatively: idx1 can be a Java array of int
             *********************************************************/
           {
               Object o_componentType=convFromRexx("O", args[1]);
               if (! (o_componentType instanceof Class))    // not a class object in hand ?
               {
                   throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                        '"'+args[0]+'"'+" componentType ["+args[1]+"] is not an instance of 'Class'!");
               }
               Class componentType=(Class) o_componentType;

               int [] indices=null;     // define a variable to hold an int-array for the indices
               int firstIdx=2;

               if (args[firstIdx].charAt(0) == '[')         // an array in hand !
               {
                   Object o_idx=convFromRexx("O", args[firstIdx]);

                   // ArrayWrapper aw=ArrayWrapper.getArrayWrapper(o_idx);  // get
                   ArrayWrapper aw=new ArrayWrapper(o_idx);  // get

                   if (aw.componentType != int.class)       // check array of indices for correct class
                   {
                       throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                            '"'+args[0]+'"'+" index array object ["+args[firstIdx]+"] not of type 'int'!");
                   }
                   indices=(int []) o_idx;                  // o.k., assign to int-array
               }
               else // individual indices, create an array of int's for it
               {
                   indices=new int [ args.length-firstIdx ];  // calculate & reserve needed array-size
                   int i=0, j=firstIdx;
                   for ( ; j<args.length; i++, j++)
                   {
                       indices[i]=new Integer(args[j]).intValue();
                   }
               }

               Object arr=Array.newInstance( componentType, indices); // create the array object
               return makeString4Rexx(arr);     // return the array object
           }


       // ==================================================================

       case ARRAY_AT:
          /*********************************************************
           * value = BSF( 'arrayAt', arrayObject, idx1 [, idx2 [, idx3 [, idx4 [, idx5]...]]] )
           *
           * - arrayObject:   array to be worked upon
           * - idx1 ... idx5 ...: index values for arrays
           *   alternatively: idx1 can be a Java array of int
            *********************************************************/
       case ARRAY_PUT:
           /*********************************************************
            * value = BSF( 'arrayPut', arrayObject, "newValue", idx1 [, idx2 [, idx3 [, idx4 [, idx5]...]]] )
            *
            * - arrayObject:   array to be worked upon
            * - newValue:      new value to set array element to
            * - idx1 ... idx5 ...: index values for arrays
            *   alternatively: idx1 can be a Java array of int
            *********************************************************/

       case ARRAY_PUT_STRICT :
           /*********************************************************
            * value = BSF( 'arrayPut', arrayObject, "argType", "newValue", idx1 [, idx2 [, idx3 [, idx4 [, idx5]...]]] )
            *
            * - arrayObject:   array to be worked upon
            * - argType:       type of "newValue"
            * - newValue:      new value to set array element to
            * - idx1 ... idx5 ...: index values for arrays
            *   alternatively: idx1 can be a Java array of int
            *********************************************************/

       case WRAP_ARRAY_OBJECT:
           /*********************************************************
            * value = BSF( 'arrayWrap', arrayObject)
            *
            * - arrayObject:   array to be worked upon
            *********************************************************/

           {
               if (args[1].charAt(0) != '[')    // not an array object!
               {
                   throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                            '"'+args[0]+'"'+" array object ["+args[1]+"] is not an array!");
               }


               Object arrayObj = mgr.lookupBean( args[1] );        // a registered array object?
               if (arrayObj == null)
               {
                   throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                            '"'+args[0]+'"'+" array object ["+args[1]+"] not registered!");
               }


               if (choice==WRAP_ARRAY_OBJECT)     // create and return the ArrayWrapper
               {
                   // return makeString4Rexx( ArrayWrapper.getArrayWrapper(arrayObj) ); // get the element and return it to Rexx
                   return makeString4Rexx( new ArrayWrapper(arrayObj) ); // get the element and return it to Rexx
               }


               if (args.length< (choice==ARRAY_AT ? 3 : 5) )   // not enough arguments?
               {
                   String errMsg = '"'+args[0]+'"'+" invalid # of args; usage: value = BSF( " +
                                    "'"+args[0]+"', arrayObject,"+
                                                   (choice==ARRAY_AT ? "" : " argType, newValue,") +
                                    " indexArray of int |or comma-separated indices| idx1 [, idx2]...";
               }

               Object o=null, o_nv=null;

               // ArrayWrapper aw=ArrayWrapper.getArrayWrapper(arrayObj); // get wrapper, so we know which component type it has
               ArrayWrapper aw=new ArrayWrapper(arrayObj); // get wrapper, so we know which component type it has

               int firstIdx=2;          // first argument with index-information, default to arrayAt()

               if (choice==ARRAY_PUT_STRICT)
               {
                   o_nv=convFromRexx( args[2], args[3] );     // get the Java object
// System.err.print("ARRAY_PUT_STRICT:  args[2]=["+args[2]+"], "+"args[3]=["+args[3]+"]: o_nv=["+o_nv+"]");
                   firstIdx=4;          // first argument with index-information for arrayPut()
               }
               else if (choice==ARRAY_PUT)      // convert Rexx argument to be stored in array to the appropriate type
               {
                   o_nv=convFromRexx(args[2]);  // get Java null or Java object from BSF registry, String else
                   Class cl=arrayObj.getClass(),
                         ct=aw.componentType;

// System.err.print("ARRAY_PUT: cl=["+cl+"] isPrimitive: ["+cl.isPrimitive()+"] "+"aw.componentType=["+ct+"] isPrimitive(): "+ct.isPrimitive()+", args[2]=["+args[2]+"] --> o_nv=["+o_nv+"]");

                   if (o_nv!=null && ct.isPrimitive() )      // if not null, determine component type of array
                   {
                       TypeConvertor tc = glob_tcr.lookup(String.class, ct);
                       o_nv=tc.convert(String.class, ct, o_nv);
                   }

                   firstIdx=3;          // first argument with index-information for arrayPut()
               }

                    // check index-information: was an array passed in or individual index-values ?
               int [] indices=null;     // define a variable to hold an int-array for the indices

               if (args[firstIdx].charAt(0) == '[')         // an array in hand !
               {
                   Object o_idx=convFromRexx("O", args[firstIdx]);

                    // get and test component type
                   // if (ArrayWrapper.getArrayWrapper(o_idx).componentType != int.class)       // check array of indices for correct class
                   if (new ArrayWrapper(o_idx).componentType != int.class)       // check array of indices for correct class
                   {
                       throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                            '"'+args[0]+'"'+" index array object ["+args[firstIdx]+"] not of type 'int'!");
                   }
                   indices=(int []) o_idx;                  // o.k., assign to int-array
               }
               else // individual indices, create an array of int's for it
               {
                   indices=new int [ args.length-firstIdx ];  // calculate & reserve needed array-size
                   int i=0, j=firstIdx;
                   for ( ; j<args.length; i++, j++)
                   {
                       indices[i]=new Integer(args[j]).intValue();
                   }
               }

               // aw=ArrayWrapper.getArrayWrapper(arrayObj);  // get
               aw=new ArrayWrapper(arrayObj);  // get

               if (choice==ARRAY_AT)
               {
                   return makeString4Rexx(aw.get(indices)); // get the element and return it to Rexx
               }
               else
               {
                   aw.put(o_nv, indices);           // put the element into the array
               }
           }
           break;



          case ARRAY_LENGTH       :
                  {
                  /* ********************************************************
                   * value = BSF( 'arrayLength', arrayObject )
                   * returns the value of the array field 'length'
                   *
                   * - arrayObject:   array to be worked upon
                   *
                   *********************************************************/
                      if (args.length!=2)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+" invalid # of args; usage: value = BSF( " +
                                  "'"+args[0]+"', arrayObject )");
                      }
                      Object arrayObj = mgr.lookupBean( args[1] );        // a registered array object ?
                      int    num      = Array.getLength( arrayObj ) ;
                      return makeString4Rexx ( new Integer(num) );
                  }
        // ==================================================================



          case EXIT               :
                  {
                  /* ********************************************************
                   * call BSF 'exit' [, return_value [, time2sleep] ]
                   * terminates the JVM; returns the string "SHUTDOWN, REXX !"
                   * - retVal : optional return value
                   * - msecs  : optional msecs to sleep before terminating the JVM
                   *********************************************************/
                      int  rc=0;                  // return code
                      long msecs2sleep=100l;      // msecs to sleep (default: 1/10 sec)

                      if (args.length > 1)
                      {
                         rc=Integer.parseInt(args[1]);    // get return code, if supplied
                                                          // get milliseconds to sleep
                         if (args.length > 2) msecs2sleep = Long.parseLong(args[2]);
                      }
                          // create and start the delayed exit thread
                      new Thread( new ExitDelayed(rc, msecs2sleep) ) .start();
                      return "SHUTDOWN, REXX !";  // indicate to Rexx what to do ... ;)
                  }
        // ==================================================================

          case GET_FIELD_VALUE           :
          case GET_FIELD_VALUE_STRICT    : // rgf, 2005-05-06

          case SET_FIELD_VALUE           :
          case SET_FIELD_VALUE_STRICT    :

          case GET_PROPERTY_VALUE        :
          case SET_PROPERTY_VALUE        :
          case SET_PROPERTY_VALUE_STRICT :
                  {

                      boolean bDebug=    // if true, show debug infos
                      false;
                      // true;
//  bDebug=true;
                      int iDebugLevel=0;

if (iDebugLevel>0)
{
    String tmp1="";
    tmp1=tmp1+"//// FIELD \\\\: args.length=["+args.length+"], [0]=["+args[0]+"]: [1]=["+args[1]+"]";
    if (args.length >= 3) tmp1=tmp1+"[2]=["+args[2]+"] ";
    if (args.length >= 4) tmp1=tmp1+"[3]=["+args[3]+"] ";
    if (args.length >= 5) tmp1=tmp1+"[4]=["+args[4]+"] ";
    if (args.length >= 6) tmp1=tmp1+"[5]=["+args[5]+"] ";

    System.err.println(tmp1);
}



                     /* ********************************************************
                      * object = BSF( 'getFieldValue',          beanName, fieldName )
                      * object = BSF( 'getFieldValueStrict',    beanName, fieldName ) // case sensitive fieldName
                      * object = BSF( 'setFieldValueStrict',    beanName, fieldName, argType, newValue )
                      * object = BSF( 'setFieldValue',          beanName, fieldName, newValue )
                      * object = BSF( 'getPropertyValue',       beanName, fieldName, index )
                      * object = BSF( 'setPropertyValueStrict', beanName, fieldName, index, argType, newValue )
                      * object = BSF( 'setPropertyValue',       beanName, fieldName, index, newValue )
                      *
                      * - beanName: the bean to lookup
                      * - fieldName: the bean to lookup
                      * - returns:  the String rendering of the bean or null, if not found
                      *********************************************************/

                     String tmp = null;        // new 2001-04-25, edited: 2003-05-10
                     String fieldName=args[2];    // store received fieldName

                          if (choice==GET_FIELD_VALUE           && args.length != 3) tmp=    "fieldName";
                     else if (choice==GET_FIELD_VALUE_STRICT    && args.length != 3) tmp=    "fieldName"; // rgf, 2005-05-06

                     else if (choice==SET_FIELD_VALUE           && args.length != 4) tmp=    "fieldName, newValue";
                     else if (choice==SET_FIELD_VALUE_STRICT    && (args.length<4 || args.length>5))   tmp= "fieldName, [newValueType,] newValue";

                     else if (choice==GET_PROPERTY_VALUE        && (args.length!=3 && args.length!=4)) tmp= "propertyName[, index]";
                     else if (choice==SET_PROPERTY_VALUE        && args.length != 5) tmp= "propertyName, index, newValue";
                     else if (choice==SET_PROPERTY_VALUE_STRICT && args.length != 6) tmp= "propertyName, index, newValueType, newValue";

                     if (tmp != null)        // wrong number of arguments
                     {
                         throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                  '"'+args[0]+'"'+": wrong number of arguments ("+args.length+");"
                                     +" usage: value=BSF("+args[0]+", beanName, " +tmp+" ).");
                     }

                     Object o = mgr.lookupBean(args[1]);

                     if (o==null)
                     {
                         throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                  '"'+args[0]+'"'+": bean '"+args[1]+"' not registered.");
                     }

                     try
                     {
                         Bean    bean  = null;
                         Integer index = null;

                         if (choice==GET_FIELD_VALUE        ||
                             choice==SET_FIELD_VALUE        ||
                             choice==GET_FIELD_VALUE_STRICT ||   // rgf, 2005-05-06
                             choice==SET_FIELD_VALUE_STRICT )
                         {           // get a field
                                // try to find any field matching the name (not case sensitive!)
                             if (choice==GET_FIELD_VALUE || choice==SET_FIELD_VALUE)
                             {
//????
if (iDebugLevel>1)
System.err.println("---> searching for fieldName (not case sensitively...)");
                                 Object tmpBean = mgr.lookupBean(args[1]);
                                 if(tmpBean==null)
                                 {
                                     // 2003-06-11, ---rgf
                                     if (args[1]==null || args[1].equals(null4Rexx))
                                     {
                                         throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                                  '"'+args[0]+'"'+": object '"+args[1]+"' not registered");
                                     }

                                     tmpBean=args[1]; // 2003-06-11, ---rgf: not in BSF-registry, assuming a String object method
//           if (bDebug) { System.err.println("invoke_strict(): assuming a String object operation desired for "+bean+"]"); }
                                 }


                                 // find correct cased field name
                                 Class c = tmpBean.getClass();    // class object for reflection
                                 if (c == Class.class)         // if already a class object, use all of its methods
                                 {
                                     c=(Class) tmpBean;
                                 }

                                       // determine correct case of method name
                                 Field[] f = c.getFields();
                                 for (int i=0; i<f.length; i++)
                                 {
                                     if(fieldName.equalsIgnoreCase(f[i].getName()))
                                     {
                                         fieldName=f[i].getName();   // remember method name
if (iDebugLevel>1)
System.err.println("---> fieldName=[" + fieldName + "]");
                                         args[2]=fieldName;         // rgf, 2005-05-06: make sure right case is now used for fieldName
                                         i=f.length;               // leave for-loop
                                     }
                                 }
//????
                             }

if (bDebug) System.err.println("--> FIELD <--(): args[0]=[" + args[0] + "] args[1]=["+args[1]+"], fieldName=["+fieldName+"]");

                             // rgf, 2005-05-07
                             // "try" the following, if not successful, try
                             // a method named "set"+fieldName with value as argument
                             // (take care of set...strict case!
                             try
                             {
                                 bean=ReflectionUtils.getField(o, fieldName);       // try to get the value
                             }
                             catch (Exception e)
                             {
                                 if (choice!=SET_FIELD_VALUE && choice!=GET_FIELD_VALUE)   //
                                 {
                                    throw (e);
                                 }

                                    // try to use a setter-method instead (fieldName led in by "set")
                                 try
                                 {
                                     if (choice==SET_FIELD_VALUE)
                                     {
if (bDebug) System.err.println("Trying 'SET"+args[2]+"' ? : " + ! (args[2].toUpperCase()).startsWith("SET"));
                                            // maybe trying a method named "get"+methName, if not already led-in with "get" ?
                                         if (! (args[2].toUpperCase()).startsWith("SET") )
                                         {
                                             String tmpArg[]={
                                                 STRING_INVOKE , // subfunction "INVOKE"
                                                 args[1],        // beanName
                                                 "SET"+args[2],  // methName ==> fieldName ?
                                                 args[3]         // argument
                                                 } ;
                                             return javaCallBSF( tmpArg );    // if it works, already prepared for Rexx
                                         }

                                     }
                                     else // choice==GET_FIELD_VALUE
                                     {
if (bDebug) System.err.println("Trying 'GET"+args[2]+"' ? : " + ! (args[2].toUpperCase()).startsWith("GET"));
                                         // maybe trying a method named "get"+methName, if not already led-in with "get" ?
                                         if (! (args[2].toUpperCase()).startsWith("GET") )
                                         {
                                             String tmpArg[]={
                                                 STRING_INVOKE , // subfunction "INVOKE"
                                                 args[1],        // beanName
                                                 "GET"+args[2]   // methName ==> fieldName ?
                                                 } ;
                                             return javaCallBSF( tmpArg );    // if it works, already prepared for Rexx


                                         }
                                     }

                                 }
                                 catch (Exception e1) { };
                             }
                         }
                         else        // get a property
                         {
                            if ( args.length>=4 && args[3]!=null && (!args[3].equals("")) )
                                     index=new Integer(args[3]);            // get Integer from String

                            bean=ReflectionUtils.getProperty(o, args[2], index);
                         }

                         if (choice==GET_FIELD_VALUE || choice==GET_FIELD_VALUE_STRICT || choice==GET_PROPERTY_VALUE)
                            return makeString4Rexx(bean.value);             // return whatever we got


                         // this section sets fields/properties
                         try         // now set the field/property to the new value
                         {
                             if (glob_tcr == null) // no TypeConvertorRegistry instance yet?
                             {
                                glob_tcr=new TypeConvertorRegistry();        // create an instance
                             }

                             if (choice==SET_FIELD_VALUE || choice==SET_FIELD_VALUE_STRICT)    // normal field
                             {
                                 if      (args.length==4 && args[3]==null) bean.value=null;  // SET_FIELD_VALUE to null
                                 else if (args.length==5 && args[4]==null) bean.value=null;  // SET_FIELD_VALUE_STRICT to null

                                 else if (choice==SET_FIELD_VALUE || args.length==4) // rgf, 2005-05-07
                                 {
                                     Object tmpValue=convFromRexx(args[3]);    // relaxed version, need to determine type
                                     if (tmpValue != null && (bean.type).isPrimitive())
                                     {
                                         // convert Rexx argument to Java
                                         TypeConvertor tc=glob_tcr.lookup(String.class, bean.type);
                                         tmpValue=tc.convert(String.class, bean.type, tmpValue);
                                     }
                                     bean.value=tmpValue;
                                 }
                                    // strict (type explicitly stated):
                                 else bean.value=convFromRexx(args[3], args[4]);

                                 ReflectionUtils.setField(o, args[2], bean, glob_tcr);


                             }
                             else
                             {
                                 if      (args.length==5 && args[4]==null) bean.value=null; // SET_PROPERTY_VALUE to null
                                 else if (args.length==6 && args[5]==null) bean.value=null; // SET_PROPERTY_VALUE_STRICT to null

                                 else if (choice==SET_PROPERTY_VALUE)
                                 {
                                     // relaxed version, need to determine type
                                     Object tmpValue=convFromRexx(args[4]);
                                     if (tmpValue != null && (bean.type).isPrimitive())
                                     {
                                         // convert Rexx argument to Java
                                         TypeConvertor tc=glob_tcr.lookup(String.class, bean.type);
                                         tmpValue=tc.convert(String.class, bean.type, tmpValue);
                                     }
                                     bean.value=tmpValue;
                                 }
                                 // strict (type explicitly stated):
                                 else bean.value=convFromRexx(args[4], args[5]);

                                 ReflectionUtils.setProperty(o, args[2], index, bean.value, bean.type, glob_tcr);
                             }

                             // return received value-string (don't know why, but keeping it for back-compatibility, --rgf, 2003-05-10
                             return args[args.length-1];
                         }
                         catch (Exception e)
                         {
//                            e.printStackTrace(System.err);

                            throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                                   '"'+args[0]+'"'+": exception while setting "
                                      +(choice==SET_FIELD_VALUE ? "field" : "property")
                                      +" ["+ args[2]+"] in object ["+args[1]+"] to ["+ (args.length>4 ? args[4] : args[3]) +"]");
                         }

                     }
                     catch (Exception e)
                     {

// System.err.println("<--------------------------------------------> begin2");
// System.err.println("<<<< baos.toString(): [" + getPrintStackAsString(e) + "] >>>>");
// System.err.println("<--------------------------------------------> end2");

                        throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                               '"'+args[0]+'"'+": exception while accessing "
                                  +( (choice==GET_FIELD_VALUE        || choice==SET_FIELD_VALUE ||
                                      choice==GET_FIELD_VALUE_STRICT || choice==SET_FIELD_VALUE_STRICT) ?
                                     "field" : "property" )
                                  +" ["+args[2]+"] in object ["+args[1]+"].");
                     }
                  }
        // ==================================================================

          case GET_STATIC_VALUE          :
          case GET_STATIC_VALUE_STRICT   : // rgf, 2005-05-06
                  {
                  /* ********************************************************
                   * call BSF 'getStaticValue', className, fieldName
                   *
                   * gets the value of a static field of the given class or interface;
                   * does *not* instantiate the class (would not be possible for
                   * abstract classes and interfaces)
                   *
                   * className: name of class or interface to lookup
                   * fieldName: name of a public static field to lookup
                   *********************************************************/
                      if(args.length!=3)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid # of args; usage: value = BSF( " +
                              "'"+args[0]+"', className, fieldName )");
                      }

                      boolean bError=false;
                      String  errTxt="",
                              fieldName=args[2]; // assign field Name


                      Class cl=null;
                      Field f=null;

                      // try   { cl=Class.forName(args[1]); }// get class object
                      // catch (ClassNotFoundException e)
                      // {
                      //    bError=true;
                      //    errTxt="class '"+args[1]+"' not found";
                      // }
                        // rgf, 2005-05-06 begin
                      Object tmpBean = mgr.lookupBean(args[1]);
                      if(tmpBean==null)
                      {
                          cl=EngineUtils.loadClass(mgr, args[1]); // ---rgf, 2003-05-10
                      }
                      else
                      {
                          // find correct cased field name
                          cl = tmpBean.getClass();    // class object for reflection
                          if (cl == Class.class)         // if already a class object, use all of its methods
                          {
                              cl=(Class) tmpBean;
                          }
                      }
                        // rgf, 2005-05-06 end


                      // cl=EngineUtils.loadClass(mgr, args[1]); // ---rgf, 2003-05-10

                          // rgf, 2005-05-06
                      if (choice==GET_STATIC_VALUE) // try to get field from case-INsensitive name
                      {
                                // determine correct case of method name
                          Field[] fa = cl.getFields();
                          for (int i=0; i<fa.length; i++)
                          {
                              if(fieldName.equalsIgnoreCase(fa[i].getName()))
                              {
                                  fieldName=fa[i].getName();   // remember method name in case sensitiveness
                                  i=fa.length;               // leave for-loop
                              }
                          }
                      }


                      if (!bError)
                      {
                         try { f = cl.getField(fieldName); }// get the field object
                         catch (NoSuchFieldException e) {
                             bError=true;
                             errTxt="class '"+args[1]+"', static field '"+args[2]+"' not found";
                         }
                      }

                      if (!bError)        // got a field object, now check whether public & static
                      {
                         int modifiers=f.getModifiers();          // get the modifiers

                         if ( (modifiers & Modifier.STATIC) == 0 )
                         {  bError=true;
                            errTxt="class '"+args[1]+"', static field '"+args[2]+"' not STATIC";
                         }
                         else             // o.k., now what?
                         {                // get the value as an Object using the class object as object
                            try { return makeString4Rexx( f.get( cl ) ); }
                            catch (IllegalAccessException e) {
//                                  e.printStackTrace(System.err);
                            }
                         }
                      }

                          // o.k., if arriving here, we're in trouble, i.e. an error occured !
                      throw new BSFException (BSFException.REASON_EXECUTION_ERROR, '"'+args[0]+'"'+": "+errTxt+"!");
                  }
        // ==================================================================

       case INVOKE_STRICT      :
                  bStrongTyping=true;       // indicate that type information is passed in by Rexx
                  {
                      boolean bDebug=false;      // if true, show debug infos
// bDebug=true;

                  /* ********************************************************
                   * value = BSF( 'invoke', [beanName], function [, argType, arg]... )
                   *
                   * - beanName: the bean whose method is to be invoked
                   * - function: the function's name
                   * - argType: the type of an argument
                   * - arg: the argument's value
                   *********************************************************/

if (bDebug) System.err.println("invokeStrict: beanName=["+args[1]+"], function: ["+args[2]+"] args.length="+args.length);

                      if((args.length<3) || ((args.length-3)%2!=0))
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+" invalid # of args; usage: value = BSF( " +
                                  "'"+args[0]+"', beanName, function [, argType, arg]... )");
                      }
                      Object bean = mgr.lookupBean(args[1]);
                      if(bean==null)
                      {
                          // 2003-06-11, ---rgf
                          if (args[1]==null || args[1].equals(null4Rexx))
                          {
                              throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                       '"'+args[0]+'"'+": object '"+args[1]+"' not registered");
                          }

                          bean=args[1]; // 2003-06-11, ---rgf: not in BSF-registry, assuming a String object method
if (bDebug) { System.err.println("invoke_strict(): assuming a String object operation desired for "+
                                 bean+"]"); }
                      }


                      //convert function-arguments to the correct type
                      Object funcArgs[] = new Object[(args.length-3)/2];
                      if(args.length>3)
                      {
                          for(int i=3, j=0; i<args.length; i+=2, j++)
                          {
                                  // convert arguments to Java type
                              funcArgs[j] = convFromRexx( args[i], args[i+1]);
                          }
                      }


                      // find correct method name
                      Class c = bean.getClass();    // class object for reflection
                      if (c == Class.class)         // if already a class object, use all of its methods
                      {
                          c=(Class) bean;
                      }

                            // determine correct case of method name
                      Method[] m = c.getMethods();
                      for (int i=0; i<m.length; i++)
                      {
                          // if(args[2].equalsIgnoreCase(m[i].getName()))
                          // rgf, 2005-05-06: do an exact match
                          if(args[2].equals(m[i].getName()))
                          {
                              args[2]=m[i].getName();   // remember method name
                              i=m.length;               // leave for-loop
                          }
                      }

                            // call the function/method
                      Object result;
                      try
                      {
                          result = EngineUtils.callBeanMethod(bean, args[2], funcArgs);
                          if (result!=null)
                          {
                             return makeString4Rexx(result);
                          }
                      }
                      catch (Exception e) {    // no method found?
//                          e.printStackTrace ();
                          System.err.println("/// Java-exception (RexxAndJava) occurred:\n["+e.getMessage()+"]\n\\\\\\");
//                          System.err.println('"'+args[0]+'"'+": got exception [" + e.getMessage ()+"].");

                             throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                                      '"'+args[0]+'"'+": object '"+args[1]+"' - method ["+args[2]+"], method not found or error (exception) executing method!" );
                      }
                  }
             break;


        // ==================================================================

       case INVOKE :
                  {
                  /* ********************************************************
                   * value = BSF( 'invoke', [beanName], function [, arg]... )
                   *
                   * - beanName: the bean whose method is to be invoked
                   * - function: the function's name
                   * - arg: the argument's value
                   *********************************************************/



                      boolean bDebug=
                                      false;
                                      // true;

if (bDebug) System.err.println("\n==> INVOKE <===");
if (bDebug) System.err.println("invoke: beanName=["+args[1]+"], function: ["+args[2]+"] args.length="+args.length);

                      int nrArgs=args.length-3; // contains nr of arguments for method
                      // Object coercedArgs[]=new Object[nrArgs];   // this array will contain the arguments co-erced to the appropriate typ
                      Object [] coercedArgs = null; // array containing the coerced arguments


                      if (args.length<3)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+" invalid # of args; usage: value = BSF( " +
                                  "'"+args[0]+"', beanName, function [, arg]... )");
                      }

                      Object bean = mgr.lookupBean(args[1]);
                      if(bean==null)
                      {
                          // 2003-06-11, ---rgf
                          if (args[1]==null || args[1].equals(null4Rexx))
                          {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": object '"+args[1]+"' not registered");
                          }
                          bean=args[1]; // 2003-06-11, ---rgf: not in BSF-registry, assuming a String object method
if (bDebug) { System.err.println("invoke(): assuming a String object operation desired for "+
                                 bean+"]"); }

                      }
if (bDebug) System.err.println("bean: ["+bean+"]");
                      //convert function-arguments to the correct type
                      // Object funcArgs[] = new Object[(args.length-3)/2];
                      Object funcArgs[] = new Object[(args.length-3)];
                      if (nrArgs>0)
                      {
                          // convert arguments to Java type, if possible
                          for (int i=3; i<args.length; i++ )
                          {
                              // funcArgs[j] = convFromRexx( args[i]);
                              funcArgs[i-3] = convFromRexx(args[i]);
                          }
                      }


if (bDebug) System.err.println("\tnrArgs=["+nrArgs+"]");
                      // find correct method name
                      Class c = bean.getClass();    // class object for reflection
                      if (c == Class.class)         // if already a class object, use all of its methods
                      {
                          c=(Class) bean;
                      }


if (bDebug) System.err.println("\tStarting class to look for method: ["+c+"]");
                        // now find a method with the needed amount of arguments
                        // start out with present class and work up to root class

                      {
                          String methNameArg=args[2],   // name as passed in (all in upper case)
                                 methName="";           // name in correct casing
                          // Object coercedArgs[]=new Object[nrArgs];   // this array will contain the arguments co-erced to the appropriate typ


                          boolean bLooking=true;
                          Class tmpClass=c, tmpSuperClass;
                          while ( bLooking )
                          {
if (bDebug) System.err.println("\t\ttmpClass: ["+tmpClass+"] ...");
                              Method[] m=tmpClass.getDeclaredMethods();

                              int i;
                              for (i=0; i<m.length && bLooking; i++)    // check out the methods
                              {
                                  methName=m[i].getName();

                                  if(methNameArg.equalsIgnoreCase(methName)) // method of this name found ?
                                  {
                                      Class paramTypes[]=m[i].getParameterTypes();
if (bDebug) System.err.println("\t\t\tmethName=["+methName+"] paramTypes.length="+paramTypes.length);
                                      if (paramTypes.length != nrArgs)  // wrong nr of arguments
                                          continue;

                                        // only access method if it is defined to be public
                                      if (!Modifier.isPublic(m[i].getModifiers()))
                                          continue ;

                                        // right number of arguments, can we coerce to them?
                                      if (nrArgs==0)     // no argument, hence found
                                                    {
                                          // FOUND !
                                          bLooking=false;            // don't look any further
                                      }
                                      else
                                      {
                                          coercedArgs=this.coerceArgs( funcArgs, paramTypes);
                                          bLooking=(coercedArgs==null); // if null, keep on searching
                                      }
if (bDebug) System.err.println("\t\tmethod was public, nrArgs matched, bLooking="+bLooking);
                                  }
                              }

                              tmpSuperClass=tmpClass.getSuperclass(); // get superclass, e.g. null, if an interface

if (bDebug) System.err.println("\t\ttmpSuperClass: ["+tmpSuperClass+"] bLooking="+bLooking+" args.length="+args.length+"...");

                              if (bLooking && (tmpClass==Object.class || tmpSuperClass==null))   // root class, method not found
                              {

                                  if (args.length==3)    // rgf, 2005-05-06, no argument given, maybe a getField intended?
                                  {
// ??????
                                      // if not a getter method, try to access a field by the name of the method
                                      if (! (args[2].toUpperCase()).startsWith("GET") )
                                      {
                                          try
                                          {
                                              String tmpArg[]={
                                                  STRING_GET_FIELD_VALUE ,
                                                  args[1],        // beanName
                                                  args[2]         // methName ==> fieldName ?
                                                  } ;
if (bDebug) System.err.println("'invoke': trying '"+STRING_GET_FIELD_VALUE+"'...");
                                              return javaCallBSF( tmpArg );    // if it works, already prepared for Rexx
                                          }
                                          catch (Exception e) { };

                                      }

// ??????
                                  }

                                    // one error could be: using a value which is a key into the BSF registry, hence give
                                  String dump="";       // programmer as much information as possible to find out about this
                                  for (i=0; i<funcArgs.length; i++)
                                  {
                                      dump=dump+"\n\t\targ # "+(i+1)+": ["+args[i+3]+
                                                "] --> ["+funcArgs[i]+"]";
                                  }
                                  throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                           '"'+args[0]+'"'+": "+
                                           "\n\tbean:   ["+bean+"]"+
                                           "\n\tmethod: ["+args[2]+"] not found!" +
                                           "\n\tCheck method name or arguments (number and types):"+dump);
                              }
                              // tmpClass=tmpClass.getSuperclass();    // check for method in superclass
                              tmpClass=tmpSuperClass;   // try the new superclass

                          }
                          args[2]=methName;// assign correct cased method name
                      }


                            // call the function/method
                      Object result;
                      try
                      {
                          result = EngineUtils.callBeanMethod(bean, args[2], coercedArgs);
                          if (result!=null)
                          {
                             return makeString4Rexx(result);
                          }
                      }
                      catch (Exception e) {    // no method found?
//                          e.printStackTrace ();
                          System.err.println("/// Java-exception (RexxAndJava) occurred:\n["+e.getMessage()+"]\n\\\\\\");
//                          System.err.println('"'+args[0]+'"'+": got exception [" + e.getMessage ()+"].");

                             throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                                      '"'+args[0]+'"'+": object '"+args[1]+"' - method ["+args[2]+"], method not found or error (exception) executing method!" );
                      }
                  }
             break;
        // ==================================================================

          case GET_VERSION        :
                  {
                  /* ********************************************************
                   * object = BSF( 'version' )
                   *
                   * - returns:  the String with the version information:
                   *             package_name package_version version_number.dateOfVersion
                   *********************************************************/

                      if (args.length!=1)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+" invalid # of args; usage: value = BSF( " +
                                  "'"+args[0]+"' )");
                      }

                      return makeString4Rexx( RAJ_VERSION );
                  }
        // ==================================================================

       case LOOKUP_BEAN        :
               {
               /* ********************************************************
                * object = BSF( 'lookup', beanName )
                *
                * - beanName: the bean to lookup
                * - returns:  the String rendering of the bean or null, if not found
                *********************************************************/

                   if (args.length!=2)
                   {
                       throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                '"'+args[0]+'"'+" invalid # of args; usage: value = BSF( " +
                                "'"+args[0]+"', beanName )");
                   }

                   Object tmp = mgr.lookupBean(args[1]);
                   if (tmp!=null)
                   {
                      return makeString4Rexx(tmp);
                   }
               }
          break;
     // ==================================================================

          case POLL_EVENT_TEXT    :
                  {
                  /* ********************************************************
                   * call BSF 'pollEventText' [timeout]
                   *
                   * timeout: time in msec to wait for receiving an eventText
                   *          before returning to Rexx (this way using Java's
                   *          locking mechanism and waiting without a busy loop)
                   *********************************************************/
                      long maxWait=0;           // wait forever
                      if (args.length>1) maxWait=Long.parseLong(args[1]); // get maxWaitTime

                      try
                         {
                            return makeString4Rexx(eventTextList.get(maxWait));
                         }
                      catch (java.lang.InterruptedException e)
                         {
//                            e.printStackTrace(System.err);
                           throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                                    '"'+args[0]+'"'+": aborting...");
                         }
                  }
        // ==================================================================

          case POST_EVENT_TEXT    :
                  {
                  /* ********************************************************
                   * call BSF 'postEventText', eventText [, priority]
                   *
                   * eventText: eventText to be added to the "eventTextList"
                   * priority:  priority level, 0=low, 1=normal (default), 2=high ("alarm level")
                   *********************************************************/
                      if(args.length>3)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid # of args; usage: value = BSF( " +
                              "'"+args[0]+"', eventText [, priority])");
                      }

                      int priority=1;     // default to normal priority

                      if (args.length==3) // priority given ?
                      {
                         priority=Integer.parseInt( args[2] );
                         if (priority<0 || priority>2) priority = 1;
                      }
                      eventTextList.put((Object) args[1], priority);
                      return args[1];
                  }
        // ==================================================================

          case REGISTER_BEAN_STRICT      :
                  bStrongTyping=true;       // indicate that type information is passed in by Rexx
                  {
                  /* ********************************************************
                   * beanName = BSF( 'registerBean', beanName, beanType [, argType, arg]... )
                   *
                   * - beanName: the bean's name, optional
                   * - beanType: the bean's type
                   * - argType: the type of a constructor argument
                   * - arg: the argument's value
                   *********************************************************/
                      if((args.length<3) || ((args.length-3)%2!=0))
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid # of args; usage: beanName=BSF( " +
                                  "'"+args[0]+"', [beanName], beanType [, argType, arg]... )");
                      }

                      //convert arguments to the correct type
                      Object funcArgs[] = new Object[(args.length-3)/2];
                      if(args.length>3)
                      {
                          char type;
                          for(int i=3, j=0; i<args.length; i+=2, j++)
                          {
                                  // convert arguments to Java type
                              funcArgs[j] = convFromRexx( args[i], args[i+1]);
                          }
                      }

                      // register the object
                      try {

                          Object bean     = EngineUtils.createBean(args[2],funcArgs);
                          String beanName = (String) convFromRexx("STring", args[1]);

                          if (beanName==null || beanName.equals(""))
                          {  // need to create a unique beanName
                             beanName=""+ bean.getClass().getName() + '@'
                                        + Integer.toHexString( bean.hashCode() );
                          }

                          // take care of ref-counters
                          int refCount    = 1;                            // ref-Counter
                          Integer i1 = (Integer) orxReferences.get( beanName );// is object already registered?
                          if (i1 != null)                                 // increase its reference counter
                          {
                             refCount = i1.intValue()+1;
                          }
                          orxReferences.put(beanName, new Integer(refCount));

                          // new object is always put into the Hashtable (replaces an existing value)
                          mgr.registerBean(beanName, bean);

                          return beanName;        // return beanName for the caller

                      }
                      catch (BSFException e)
                      {
//                          e.printStackTrace ();

                          throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                                   '"'+args[0]+'"'+": could not create a bean from instantiating '"+
                                   args[2]+"'.");
                      }
                  }
        // ==================================================================

        // ==================================================================
          case REGISTER_BEAN      :
                  {
                  /* ********************************************************
                   * beanName = BSF( 'registerBean', beanName, beanType [, arg]... )
                   *
                   * - beanName: the bean's name, optional
                   * - beanType: the bean's type
                   * - arg: the argument's value
                   *********************************************************/
                      if (args.length<3)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid # of args; usage: beanName=BSF( " +
                                  "'"+args[0]+"', [beanName], beanType [, argType, arg]... )");
                      }

                      Class cl=EngineUtils.loadClass(mgr, args[2]);

                      Constructor [] constr=cl.getConstructors();
                      if (constr.length==0)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": no constructors available for Java class ["+args[2]+"], cannot create an instance...\n");
                      }


                      int nrArgs=args.length-3; // arguments given for creating object
                      Object funcArgs[] = new Object[nrArgs] ;
                      Object [] coercedArgs = null;

                      // check Rexx arguments, replace nulls with Java nulls and
                      // references to the BSF registry with the Java object
                      if(args.length>3)
                      {
                          for(int i=3; i<args.length; i++)
                          {
                                  // convert nulls and beans to Java null and Java object
                              funcArgs[i-3] = convFromRexx( args[i] );
                          }
                      }

                      boolean bFound=false;
                        // find matching constructor
                      int i=0;
                      for (i=0; i<constr.length; i++)
                      {
                          if (!Modifier.isPublic(constr[i].getModifiers()))  // not a public constructor
                              continue ;

                          Class paramTypes[]=constr[i].getParameterTypes();
                          if (paramTypes.length != nrArgs) continue;    // nr of arguments does not match

                          if (nrArgs>0)
                             coercedArgs=this.coerceArgs( funcArgs, paramTypes);

                          if (nrArgs==0 || coercedArgs != null)
                          {
                              bFound=true;
                              break;
                          }
                      }

                      if (!bFound)  // no matching constructor found
                      {
                            // one error could be: using a value which is a key into the BSF registry, hence give
                          String dump="";       // programmer as much information as possible to find out about this
                          for (i=0; i<funcArgs.length; i++)
                          {
                              dump=dump+"\n\t\targ # "+(i+1)+": ["+args[i+3]+
                                        "] --> ["+funcArgs[i]+"]";
                          }
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": no public constructor matching given arguments found for Java class ["+
                                   args[2]+"], cannot create an instance...\n"+
                                  "\n\tCheck method name or arguments (number and types):"+dump);
                      }

                      Object bean=null;
                      // create and register the object
                      try {
                          // Object bean     = EngineUtils.createBean(args[2],funcArgs);
                          bean     = constr[i].newInstance(coercedArgs);
                      }
                      catch (Exception e)   // 2003-05-09, --rgf, maybe an IllegalAccessException?
                      {
                          boolean bRaise=true;      // by default raise exception

                          if (e instanceof IllegalAccessException &&
                              bMethodHasSetAccessible &&
                              Modifier.isPublic(constr[i].getModifiers())   )   // if a public method allow access to it
                          {
                              constr[i].setAccessible(true);    // allow unconditional access
                              bRaise=false;     // assume everything goes o.k.

                              try
                              {
                                  bean = constr[i].newInstance( coercedArgs );  // create instance
                              }
                              catch (Exception e2)  // puh, this is a tough one !
                              {
                                  bRaise=true;  // now, raise exception
                                  e=e2;         // give information about this one
                              }
                          }

                          if (bRaise)   // do we have to raise an exception ?
                          {
                              // re-throw the exception
                              throw new BSFException (BSFException.REASON_OTHER_ERROR,
                                                      '"'+args[0]+'"'+": could not create a Java object from instantiating '"+
                                                      args[2]+" with the given arguments'.\n"+
                                                      "\t\tusing constructor: ["+constr[i]+"]\n"+
                                                      "\t\t raised exception: ["+e+"]");
                          }
                      }


                        // creating went well, now registering new Java object
                      String beanName = ( convFromRexx(args[1])==null? null : args[1]);

                      if (beanName==null || beanName.equals(""))
                      {  // need to create a unique beanName
                         beanName=""+ bean.getClass().getName() + '@'
                                    + Integer.toHexString( bean.hashCode() );
                      }

                      // take care of ref-counters
                      int refCount    = 1;                            // ref-Counter
                      Integer i1 = (Integer) orxReferences.get( beanName );// is object already registered?
                      if (i1 != null)                                 // increase its reference counter
                      {
                         refCount = i1.intValue()+1;
                      }
                      orxReferences.put(beanName, new Integer(refCount));

                      // new object is always put into the Hashtable (replaces an existing value)
                      mgr.registerBean(beanName, bean);

                      return beanName;        // return beanName for the caller
                  }
        // ==================================================================

          case UNREGISTER_BEAN    :
                  {
                  /* ********************************************************
                   * call BSF 'unregisterBean', beanName
                   *
                   * - beanName: the bean to unregister
                   *********************************************************/
                      if(args.length!=2)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid # of args; usage: call BSF " +
                              "'"+args[0]+"', beanName");
                      }

                      if (unregisterBean(args[1]))  // unregister bean, adjust ref-counter
                      {
                          return args[1];           // return the bean
                      }

                  }
             break;
        // ==================================================================

          case SET_REXX_NULL_STRING :
                  {
                  /* ********************************************************
                   * call BSF 'setRexxNullString', newString
                   *
                   * - newString: string to use for indicating "null" to/by Rexx
                   *              (default is the string ".NIL" in field "null4Rexx"
                   *********************************************************/
                      if(args.length!=2 || args[1]==null)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid arguments; usage: call BSF " +
                              "'"+args[0]+"', newString");
                      }
                      null4Rexx=args[1];        // set new value
                      return args[1];
                  }


        // ==================================================================

          case SLEEP :  // meant for Regina and other Rexxes which don't have a SysSleep() by default
                  {
                  /* ********************************************************
                   * call BSF 'sleep', time2sleep
                   *
                   * - time2sleep: decimal number of seconds to sleep, e.g. '10' (ten seconds),
                   *               '0.5' (half a second), '0.01' (1/100 second), '600' (10 minutes)
                   *********************************************************/

                      if(args.length!=2 || args[1]==null)
                      {
                          throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": invalid arguments; usage: call BSF " +
                              "'"+args[0]+"', time2sleep");
                      }

                      long sleepTime = 0l;

                      try {   // convert received string into a long (rounded up)
                              sleepTime = (long) ((new java.math.BigDecimal(args[1])).doubleValue()*1000+0.5d);
                          }
                      catch (Exception e)
                          {
                              throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                   '"'+args[0]+'"'+": argument 'time2sleep', value '"+args[1]+"' not numeric");
                          }

                      Thread thisThread = Thread.currentThread(); // get current Thread
                      try
                      {
                         thisThread.sleep( sleepTime );   // sleep the indicated time
                      }
                      catch (Exception e)
                      {
                         e.printStackTrace();
                      }

                      break;
                  }




        // ==================================================================

           case WRAP_ENUMERATION_OBJECT:
               /*********************************************************
                * value = BSF( 'wrapEnumeration', arrayObject)
                *
                * - arrayObject:   array to be worked upon
                *********************************************************/
               {

                   Object enumObj = mgr.lookupBean( args[1] );        // a registered object?
                   if (enumObj == null)
                   {
                       throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                '"'+args[0]+'"'+" object ["+args[1]+"] not registered!");
                   }

                   if (!(enumObj instanceof Enumeration))
                   {
                       throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                '"'+args[0]+'"'+" object ["+args[1]+"] does not have the 'Enumeration' interface implemented!");
                   }
                   return makeString4Rexx( new EnumerationWrapper( (Enumeration) enumObj) ); // wrap the object and return it to Rexx
               }



        // ==================================================================
          default:
                  throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                           "'"+args[0]+"': unknown function name (no such function implemented).");
       }
        // ==================================================================


        return null4Rexx;       // return the Rexx string representing explicitly "null"
    }



     // rgf, 2005-05-07
    /** Returns <code>printStackTrace()</code> as a <code>String</code>.
     *
      * @param e an Exception object.
      * @return  a String
    */
    private String getPrintStackAsString(Exception e)
    {
       ByteArrayOutputStream baos = new ByteArrayOutputStream();
       PrintStream           ps   = new PrintStream(baos);
       e.printStackTrace(ps);
       return baos.toString();
    }





    /** Converts the string from Rexx to the indicated type. If
      * a type 'O' is indicated and the value for it is an empty string,
      * then <code>null</code> is returned for it.

      * @param type one of
                    &quot;<strong>BO</strong>olean&quot; (i.e. '0' or '1'),
                    &quot;<strong>BY</strong>te&quot;,
                    &quot;<strong>C</strong>har&quot;,
                    &quot;<strong>D</strong>ouble&quot;,
                    &quot;<strong>F</strong>loat&quot;,
                    &quot;<strong>I</strong>nt&quot;,
                    &quot;<strong>L</strong>ong&quot;,
                    &quot;<strong>O</strong>bject&quot; (Object, i.e. a registered bean),
                    &quot;<strong>SH</strong>ort&quot;, and
                    &quot;<strong>ST</strong>ring&quot;.

      * @param value the value from Rexx (a string) to be converted to the
               Java type.
      *
      * @return  the converted value.
      */
    private Object convFromRexx(String type, String value) throws BSFException
    {
        if (value==null) return null;

         if (value.equals(null4Rexx))   // Rexx-String indicating "null" ?
            return null;


         switch (Character.toUpperCase(type.charAt(0)) )
         {

             case 'S':
                        if (type.length() == 1) break;    // leave switch
                        switch (Character.toUpperCase(type.charAt(1)))
                        {
                           case 'T':    // String
                                     return value;

                           case 'H':    // Short
                                     return new Short( value );

                           default:  break;     // unknown type
                        }

             case 'O':  // no object indicated, hence return null
                        if (value.equals("")) return null;

                        Object tmp;
                        tmp = mgr.lookupBean(value);

// System.err.println("===> RAJ()-convFromRexx: type="+pp(type)+", value="+pp(value)+" -> tmp="+pp(tmp));

                        if (tmp==null){
                            throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
                                "Bean ["+value+"] not registered.");
                        }
                        return tmp;

             case 'I':  return new Integer      (value);
             case 'C':  return new Character    (value.charAt(0));

             case 'B':  // need second character
                        if (type.length() == 1) break;    // leave switch
                        switch (Character.toUpperCase(type.charAt(1)))
                        {
                           case 'O':    // Boolean
                                     return new Boolean( value.equals("1") );

                           case 'Y':    // Byte
                                     return new Byte( value );

                           default:  break;     // unknown type
                        }


             case 'F':  return new Float        (value);
             case 'L':  return new Long         (value);
             case 'D':  return new Double       (value);

         }
         throw new BSFException (BSFException.REASON_INVALID_ARGUMENT,
             "unknown argType ["+type+"], with value ["+value+"].");
    }




    /** Replaces a reference to a Java object with the Java object.
     *
     * @param value this String is used to look up the BSF registry. If an entry is found (a
     * Java object) it will get returned, if not (or the value is <code>null</code>) then
     * <code>null</code> is returned.
     *
     * @return the Java <code>object</code> stored in the BSF registry, <code>null</code> else
     **/
    private Object convFromRexx(String value) throws BSFException
    {
         if (value==null) return null;

         if (value.equals(null4Rexx))   // Rexx-String indicating "null" ?
            return null;

         // Is this a reference to a Java object in the BSF registry?
         // If so, return Java object
         Object tmp=mgr.lookupBean(value);
         if (tmp==null) return value;
         return tmp;
    }


    /** Coerces the arguments from Rexx to the appropriate Java types.
     * @param funcArgs an <code>Object</code> array (actually the Rexx arguments of type
     *        <code>String</code>)
     * @param paramTypes an <code>Class</code> array, containing the Java class objects to which
     *        the Rexx arguments have to be converted to
     * <!-- (rgf, 2003-05-08) -->
     *  @return an array with the coerced arguments, if successful, <code>null</code> else.
     */
    Object [] coerceArgs(Object [] funcArgs, Class [] paramTypes)
    {
        int k,
            nrArgs=funcArgs.length;

        boolean bDebug=
             false;
             // true;

if (bDebug) System.err.println("\n---   ---   ---   ---   ---");

        Object [] coercedArgs = new Object [nrArgs];

        for (k=0; k<nrArgs;k++)
        {
if (bDebug) System.err.println("\tcoerceArgs() ---> k="+k+" (funcArgs.length="+funcArgs.length+
                                              ", paramTypes.length="+paramTypes.length+")");

if (bDebug) System.err.println("\t\tfuncArgs[k]=["+funcArgs[k]+"] ---> ["+paramTypes[k]+"]");

            if (funcArgs[k]==null)    // assuming explicit null is o.k. in any case
            {
if (bDebug) System.err.println("\t           ) ---> k=["+k+"]: null!");
                coercedArgs[k]=null;
            }
                  // if a Java object in hand, if so compatible with paramType ?
//            else if (!(funcArgs[k] instanceof String))

                // rgf, 2005-06-18, special case found: an interface is given, that is available for String as well!
            else if ( paramTypes[k].isInterface()  || (!(funcArgs[k] instanceof String)))
            {

if (bDebug) System.err.print("\t           ) ---> k="+k+": an Object not a String in hand: funcArgs ["+
                                          funcArgs[k]+"] to paramTypes ["+paramTypes[k]+"], assignable?");
// rgf, 2005-10-32                if (!(paramTypes[k].isAssignableFrom (funcArgs[k].getClass()))) // not compatible, break
if (bDebug)
{
    if (paramTypes[k].isArray()) {
        System.out.println("\n\t\t==> ==> ["+paramTypes[k]+"]: paramTypes[k].isArray=true, class=["+paramTypes[k].getClass()+"]");

        // System.out.println("\t\tgetLength(): "+((Array)paramTypes[k]).getLength());
        System.out.println("\t\t==> ==> ["+funcArgs[k]+"]: funcArgs[k].getClass()="+funcArgs[k].getClass());


System.out.println("paramTypes[k] instanceof (String): "+paramTypes[k] instanceof String);
System.out.println("funcArgs[k] instanceof (String):   "+funcArgs[k] instanceof String);
ArrayWrapper paw=new ArrayWrapper(paramTypes[k]);
ArrayWrapper faw=new ArrayWrapper(funcArgs[k]);

System.out.println("paw=["+paw+"], dimensions="+paw.dimensions+", componentType=["+paw.componentType+"]");
System.out.println("faw=["+faw+"], dimensions="+faw.dimensions+", componentType=["+faw.componentType+"]");

    }
}

                if (!(paramTypes[k].isAssignableFrom (funcArgs[k].getClass()))) // not compatible, break
                {
if (bDebug) System.err.println(" NO ! ");

                    break;
                }
if (bDebug) System.err.println(" yes. ");

                coercedArgs[k]=funcArgs[k];
if (bDebug) System.err.println("\t           ) ---> k="+k+": successful coercion.");
                continue;
            }
            else
            {
                String tmpArg=(String) funcArgs[k];  // ---rgf, for {B|b}oolean coercion

                  // do we need a String, if so, we have it already!
                if (paramTypes[k]==String.class) {
                    coercedArgs[k]=funcArgs[k];
if (bDebug) System.err.println("\t           ) ---> k="+k+": successful coercion to String.");
                    continue;
                }

                  // try to co-erce argument
                TypeConvertor cvtor = glob_tcr.lookup (String.class, paramTypes[k]);
                if (cvtor==null)   // no convertor found, break; look for another method
                {
if (bDebug) System.err.println("\t           ) ---> k="+k+": no cvtor found! paramTypes[k]: ["+paramTypes[k]+")");
                    break;
                }


                if (paramTypes[k]==boolean.class || paramTypes[k]==java.lang.Boolean.class)   // ---rgf, 2005-07-07: make sure Boolean are converted correctly
                {
if (bDebug) System.err.println("\t  >>> >>>  checking for {B|b}oolean value!");
                    if (((String)funcArgs[k]).compareTo("1")==0 ) // || ((String)funcArgs[k]).compareToIgnoreCase("TRUE")==0 )
                    {
                        tmpArg="true";
                    }

                }

                    // 2003-09-10, make sure to intercept exceptions while converting
                    // Object o= cvtor.convert( String.class, paramTypes[k], funcArgs[k]);
                Object o=null;
                try
                {
                    // o=cvtor.convert( String.class, paramTypes[k], funcArgs[k]);
                    o=cvtor.convert( String.class, paramTypes[k], tmpArg);
if (bDebug) System.err.println("\t           coercion succssful, o=["+o+"], type: ["+o.getClass()+"]. <--- <--- <---");

                }
                catch (Exception e)
                {
                   o=null;
                }


                if (o==null) // co-ercion failed: this is the wrong method, hence break!
                {
if (bDebug) System.err.println("\t           ) ---> k="+k+": coercion yielded 'null', breaking!!");
                   break;
                }
                coercedArgs[k]=o; // save coerced object instead
            }
        }

if (bDebug) System.err.println("\tcoerceArgs() ---> k="+k+" nrArgs="+nrArgs);

        if (k==nrArgs)    // no break, hence all arguments co-erced successfully!
        {
if (bDebug) System.err.println("\tcoerceArgs(): SUCCESSFUL! :-)");
            return coercedArgs;

        }

if (bDebug) System.err.println("\tcoerceArgs(): UNsuccessful :-( ---> returning 'null'!");
        return null;        // coercion was *not* successful, hence return nothing
    }






    /** This method allows for unregistering Beans, taking into account reference counters.
      * Only if the reference counter is 0, will the given Bean be unregistered. (2003-01-06, ---rgf)
      *
      *  @param beanName name of the Bean to be unregistered
      *
      *  @return returns <code>true</code>, if the Bean exists, <code>false</code> else.
      */
    protected boolean unregisterBean(String beanName )
    {
        // take care of ref-counters
        Integer i1 = (Integer) orxReferences.get(beanName);    // get refCount-value
        if (i1 != null)                                     // decrease its reference counter
        {
           int refCount = i1.intValue()-1;  // get and decrease refCount

           if (refCount > 0)                // still references on the (Object) Rexx side
           {
               orxReferences.put(beanName, new Integer(refCount));
           }
           else                             // no more references on the (Object) Rexx side
           {                                // safe to remove Java object
               orxReferences.remove(beanName);
               mgr.unregisterBean(beanName);
           }
           return true;
        }

        return false;
    }




    /** This method determines what String content to return.
      *
      *  @param o the object which should get returned as a string to Rexx.
      *
      *  @return  a String rendering of the primitive types and String.
      *           Otherwise create a unique bean name, register the object
      *           under it and return the bean name. If <code>null</code>,
      *           then the string of {@link #null4Rexx} is returned to Rexx.
      */
    protected String makeString4Rexx(Object o)  // changed 2003-01-06 from "makeResultString()" to "makeString4Rexx()"
    {
       if (o==null)
          return null4Rexx;                     // return the Rexx-String representing "null"

       Class cl = o.getClass();

       if (cl==String.class)
       {
          return (String) o;
       }

       else if (cl==Boolean.class )
       {
          return ( ((Boolean) o).booleanValue() ? "1" : "0");     // return boolean value
       }


       else if (    cl.isPrimitive() ||
                cl==Integer.class    || cl==Character.class  ||
                cl==Long.class       ||
                cl==Float.class      || cl==Double.class     ||
                cl==Byte.class       || cl==Short.class
               )
       {

          return o.toString();  // o.k.
       }

                // an Object, register it as a bean, so it becomes
                // possible to refer to from Rexx as well
       String beanName = "" + o.getClass().getName() + '@'
                            + Integer.toHexString( o.hashCode() );

       Object tmp = mgr.lookupBean( beanName );         // already defined?

       // take care of ref-counters
       int refCount    = 1;                             // ref-Counter
       Integer i1 = (Integer) orxReferences.get( beanName ); // is object already registered?
       if (i1 != null)                                  // increase its reference counter
       {
          refCount = i1.intValue()+1;
       }
       orxReferences.put(beanName, new Integer(refCount));

       if (tmp != null) return beanName;                // if so, return beanName

       mgr.registerBean( beanName, o ); // register the object under this new beanName
       return beanName;                 // return beanName for the caller
    }




   /** Class to exit the JVM, will be executed concurrently, sleeps and then
     * <code>calls System.exit(exit_number)</code>. This allows the Rexx side
       to gain time to shutdown properly.
     */
   class ExitDelayed implements Runnable
   {
         /** Holds the return value to be returned.             */
         int exitCode=0;

         /** Holds the time in milliseconds to wait before terminating the virtual machine.
             This should allow Rexx time to shutdown properly.  */
         long sleepTime=100l;

         /** Saves the <code>exitCode</code>. */
         ExitDelayed (int exitCode, long sleepTime)
         {
            this.exitCode  = exitCode;  // save exitCode
            this.sleepTime = sleepTime; // save time to sleep before terminating the JVM
         }

         /** This makes this thread runnable. Will wait {@link #sleepTime} and
           * terminates the Java virtual machine with {@link #exitCode}.
           */
         public void run() // throws InterruptedException
         {
            Thread thisThread = Thread.currentThread(); // get current Thread
            try
            {
               mgr.terminate();         // let the BSF manager "terminate gracefully"
               thisThread.sleep( sleepTime );   // sleep the indicated time
            }
            catch (InterruptedException e)
            {
               e.printStackTrace();
            }
            finally                     // in any case, kill the JVM!
            {
              System.exit(exitCode);    // now kill JVM
            }
         }
   }




    /** Inner class to implement a Synchronized ReaderWriter, where a
      * read on an empty Vector blocks the read, until a write occurs (with optional timeout).
      */
    protected class SyncPutGet
    {
        /** Vector to store the text sent by the eventListeners. */
        private Vector eventTextVector     = new Vector();      // normal priority
        private Vector eventTextVectorLow  = new Vector();      // low ("batch") priority
        private Vector eventTextVectorHigh = new Vector();      // high ("alarm") priority


        /** Get an object from the vector. If empty, block for the
          * given time.
          *
          * @param millis time to wait in milliseconds, if 0 waits until an object becomes available.
          *
          * @return the first object from the vector or <code>null</code>, if
          *         empty.
          */
        synchronized Object get (long millis) throws java.lang.InterruptedException
        {
            if ( eventTextVector    .isEmpty() &&
                 eventTextVectorLow .isEmpty() &&
                 eventTextVectorHigh.isEmpty()    )     // empty, block until filled
            {
               wait(millis);
            }

            if (eventTextVector    .isEmpty() &&
                eventTextVectorLow .isEmpty() &&
                eventTextVectorHigh.isEmpty()    )     // empty, return null
            {
// System.err.println("RexxAndJava.SyncPutGet.get(), about to return, nothing in any queue.");
                return null;    // awakened due to time-out?
            }

            Object o;
                // return the eventText in the order: High-Normal-Batch
            if ( ! eventTextVectorHigh.isEmpty())       // Highest priority!
            {
                // for supporting JRE 1.1.8, ---rgf, 2001-05-26
               o=eventTextVectorHigh.firstElement();
                 eventTextVectorHigh.removeElementAt(0);
            }

            else if ( ! eventTextVector.isEmpty() )     // Normal priority
            {
                // for supporting JRE 1.1.8, ---rgf, 2001-05-26
               o=eventTextVector.firstElement();
                 eventTextVector.removeElementAt(0);
            }

            else   // default: now lowest priority's turn
            {
                // for supporting JRE 1.1.8, ---rgf, 2001-05-26
               o=eventTextVectorLow.firstElement();
                 eventTextVectorLow.removeElementAt(0);
            }

            return o;
        }

        /** Adds an object to the vector and makes sure that waiting threads
          * are notified.
          *
          * @param o object to add to the vector.
          *
          * @param priority an integer value indicating the priority of the posted
          *        event, one of:

          *        <p><ul>
                   <li>&quot;<code>2</code>&quot;: high ('alarm level') priority.
                   <li>&quot;<code>1</code>&quot;: normal (<strong>default</strong>) priority, and
                   <li>&quot;<code>0</code>&quot;: low ('batch level') priority,
                   </ul>
          *
          */
        synchronized void put (Object o, int priority)
        {
            switch (priority)
            {
               case  2:         // add the element to the high priority vector
                           eventTextVectorHigh.addElement(o);   // changed to addElement() for JRE 1.1.8, ---rgf
                  break;

               case  0:         // add the element to the low priority vector
                           eventTextVectorLow.addElement(o);    // changed to addElement() for JRE 1.1.8, ---rgf
                  break;

               case  1:         // add the element to the normal priority vector
               default:
                           eventTextVector.addElement(o);       // changed to addElement() for JRE 1.1.8, ---rgf
                  break;
            }

            notifyAll();        // now notify all
        }
    }

    /** 'Pretty print': returns string value of supplied object enclosed in square brackets. */
    String pp(Object o)
    {
        if (o==null) return "<null>";
        return "["+o+"]";

    }
}
/*
javah -jni  org.rexxla.bsf.engines.rexx.RexxAndJava
javap -s -p org.rexxla.bsf.engines.rexx.RexxAndJava
*/
