package org.rexxla.bsf.engines.rexx;

import java.util.*;
import java.io.*;
import com.ibm.bsf.*;
import com.ibm.bsf.util.BSFEngineImpl;
import org.rexxla.bsf.engines.rexx.RexxAndJava;


/**
  * This class manages the calls from Java via JNI to Rexx or Object Rexx
  * using the RexxSAA interface.
  *
  * Proof of concept by Peter Kalender (Oct 2000 to Feb 2001, student of Rony G. Flatscher
 *  at the University of Essen).
  *
 * <pre>------------------------ Apache Version 2.0 license -------------------------
 *    Copyright (C) 2001-2006 Rony G. Flatscher
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        <a href="http://www.apache.org/licenses/LICENSE-2.0">http://www.apache.org/licenses/LICENSE-2.0</a>
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 * ----------------------------------------------------------------------------- </pre>
  *
  * @version 2.6, 2006-01-01
  * @author Rony G. Flatscher (<a href="http://www.wu-wien.ac.at">WU-Wien/Wirtschaftsuniversit&auml;t Wien</a>, <a href="http://www.wu-wien.ac.at/english">http://www.wu-wien.ac.at/english</a>)
  *
  */


/*
     2005-12-28, ---rgf, changed license to Apache
 */
public class RexxEngine extends BSFEngineImpl {
    /** Version string indicating version of this class (majorVersion*100+minorVersion
     *  concatenated with a dot and the sorted date of last change.
     */
    static public String version = "206.20060101";


    private boolean bDebug= false ; // 20040404, ---rgf

    /** Allows using the Rexx interpreter from Java. */
    // private RexxAndJava   rexxInterface;
    private RexxAndJava   rexxInterface;  // allow access to package programs
    protected RexxAndJava getRexxInterface()
    {
        return rexxInterface;
    }

    /** Stores the script code supplied with eval(), exec(), apply(), in order to be
     *  able to locate the code in call(). [Hint: Xalan as per 2002-09 uses BSF in
     *  its extension mechanism such, that it first calls exec() to have the script
     *  pre-compiled and thereafter uses call() to invoke the script.
     */
    private String lastScriptCode = null;

    /** Initializes the Rexx engine and causes the external
      * functions 'BSF' and 'BSFPollEvents' to be registered with the
      * Rexx interpreter. Therefore Rexx programs invoked from Java
      * have these two functions available to them and are therefore
      * able to communicate with Java.
      */
    public void initialize (BSFManager mgr, String lang,
                            Vector declaredBeans) throws BSFException
    {
        super.initialize (mgr, lang, declaredBeans);
        rexxInterface = new RexxAndJava(mgr, this);
        rexxInterface.jniRegisterBSF();
    }



    /** Evaluates (executes) a piece of Rexx code
      * (can be a whole program as well)
      * <em>without</em> any Java-supplied arguments and returns the result.
      *
      * @param source if set to {@link RexxAndJava#EVENT_TEXT},
      *               then <code>oscript</code> is put into the event queue
      *               to be polled by Rexx. If this text starts with the
      *               string <code>&quot;&#47;*2*&#47;&quot;</code> it is
      *               put into the alarm (highest dispatch) level vector, if it starts with the
      *               string <code>&quot;&#47;*0*&#47;&quot;</code> it is
      *               put into the batch (lowest dispatch) level vector,
      *               else it is put into the normal level vector.
      *
      * @param lineNo not used.
      * @param columnNo not used.
      * @param oscript the String version of this object [using toString()]
      *                represents the Rexx code to be executed.
      *                Its statements are delimited with a ';' or a
      *                CR (carriage-return) and/or LF (line-feed).
      *
      * @return a String object with the result from Rexx or <code>null</code>.
      */
    public Object eval (String source, int lineNo, int columnNo,Object oscript)
                                        throws BSFException
   {
if (bDebug) System.err.println("RexxEngine.eval() - begin...");

        String script = oscript.toString ();

        if(source.equals(RexxAndJava.EVENT_TEXT))     // dispatching eventListener text
        {
           String tmp = script.substring(0, 5);

// System.err.println("}-}-} RexxEngine.eval(), source=["+source+"]\n\tscript=["+script+"]");

            if (tmp.equals("/*2*/"))                    // alarm (highest) level
            {
                rexxInterface.eventTextList.put(script, 2);
            }
            else if (tmp.equals("/*0*/"))               // batch (lowest) level
            {
                rexxInterface.eventTextList.put(script, 0);
            }
            else                                      // normal level (default)
            {
                rexxInterface.eventTextList.put(script, 1);       // put text into normal level event  vector
            }

            return null;
        }
        else
        {
           lastScriptCode=script;       // save script's code, fall 2002, ---rgf: adjusts for some usage patterns

// System.err.println("RexxEngine.eval() - before: rexxInterface.jniRexxStart (...)");

           Object obj=rexxInterface.jniRexxStart (script, null); // ---rony (now has a return value)

// System.err.println("RexxEngine.eval() - after : rexxInterface.jniRexxStart (...)");

if (bDebug)
{

System.err.println("RexxEngine.eval() - end, returning ["+obj+"]...");
System.err.println("\trexxInterface.jniRexxDidRexxTerminate()="+rexxInterface.jniRexxDidRexxTerminate());
System.err.println("\trexxInterface.jniRexxWaitForTermination()="+rexxInterface.jniRexxWaitForTermination());
System.err.println("\trexxInterface.jniRexxDidRexxTerminate()="+rexxInterface.jniRexxDidRexxTerminate());
System.err.println("\tnow RETURNING...");
}
           return obj;
        }
    }




    /** Calls (executes) a Rexx program which code has been supplied before via an invocation of
     *  &quot;apply()&quot;,
     *  &quot;exec()&quot; or &quot;eval()&quot; with the Java-supplied arguments and returns
     *  the result.
     *  <p>[Hint: This follows after learning how Xalan is employing BSF: it first calls &quot;exec()&quot;
     *  and then &quot;call()&quot;, hence &quot;call()&quot; does <em>not</em> receive the source
     *  to be executed.]
      *
       @deprecated            &quot;

       This <code>call(...)</code> method may be enhanced in the future, once an openly
       defined and available interface to the OO-part of Object Rexx is made
       available by IBM. Then the argument &quot;<code>object</code>&quot; would denote
       an Object Rexx object, the argument &quot;<code>name</code>&quot; the name of one of its methods
       and the argument &quot;<code>args</code>&quot; the arguments for the method.


       <br> <strong>If possible at all, use method
    {@link #apply(String source, int lineNo, int columnNo, Object oscript, java.util.Vector vArgNames, java.util.Vector vArgs ) apply(...)}
       instead.</strong>
      *
      * @param object not used.
      * @param name   not used.
      * @param args arguments to be passed to Rexx.
      *
      * @return a String object with the result from Rexx or <code>null</code>.
      */
    public Object call (Object object, String name, Object[] args)
                        throws BSFException {

if (bDebug) System.err.println("RexxEngine.call() - begin...");

        // String stringArg[] = new String [ args.length ];
        String stringArg[] = new String [ (args!= null ? args.length : 0) ];


        // 2003-01-06, ---rgf, deal with arguments (non-primitive Java objects) which need to be turned into Beans
        Vector beanVector = new Vector(args.length);   // maximum size
        Object [] res;                  // to contain the results of args2RexxString()


        // for (int i=0 ; i < args.length ; i++)   // turn argument objects into string arguments for Rexx
        for (int i=0 ; i < (args != null ? args.length : 0) ; i++)   // turn argument objects into string arguments for Rexx
        {
           if (args[i]!=null)
           {
        //      stringArg[i]=args[i].toString();
              res = args2RexxString( args[i] );         // get String or Bean for this argument
              if (((Boolean) res[0]).booleanValue())    // a Bean had to be created
              {
                  beanVector.addElement(res[1]);        // remember beanName for later deregistering
              }
              stringArg[i]=(String) res[1];             // String value for Rexx
           }
           else stringArg[i]=null;
        }

        if (bDebug) {    // defined as protected in superclass BSFEngineImpl
            System.err.println("RexxEngine.call(): do not use this method, use RexxEngine.apply(...) instead!"+
               " This method will get enhanced, once the OO-interface to Object Rexx is made available by IBM." +
               " (Hint: this method only works, if apply(), eval() or exec() was called beforehand, "+
               " in order to get a hold of the source code.)" );
        }

            // lastScriptCode only contains code, if apply(), eval() or exec() was called already

// System.err.println("RexxEngine.call() - before: rexxInterface.jniRexxStart (...)");

        Object o= rexxInterface.jniRexxStart (lastScriptCode, stringArg);

// System.err.println("RexxEngine.call() - after : rexxInterface.jniRexxStart (...), before terminating Rexx.");

        // now deregister Beans
        Enumeration e=beanVector.elements();    // get elements for Enumeration

        while ( e.hasMoreElements() )
        {
            rexxInterface.unregisterBean( (String) e.nextElement());
        }

// System.err.println("RexxEngine.call() - after : deregistering Beans, before terminating Rexx.");

if (bDebug) System.err.println("RexxEngine.call() - end, returning ["+o+"]...");
       return o;
    }




    /** Applies (evaluates) a piece of Rexx code
      * (can be a whole program as well)
      * <em>with</em> Java-supplied arguments and returns the result.
      *
      * @param source not used.
      * @param lineNo not used.
      * @param columnNo not used.
      * @param oscript the String version of this object [using toString()]
      *                represents the Rexx code to be executed.
      *                Its statements are delimited with a ';' or a
      *                an LF (line-feed, &quot;0x0a&quot;), preceeded by an
                       optional CR (carriage-return, &quot;0x0d&quot;).

      * @param vArgNames not used.
      * @param vArgs arguments to be passed to Rexx.
      *
      * @return a String object with the result from Rexx or <code>null</code>.
      */
    public Object apply (String source, int lineNo, int columnNo,
                         Object oscript,
                         java.util.Vector vArgNames,
                         java.util.Vector vArgs
                        )
                        throws BSFException {

if (bDebug) System.err.println("RexxEngine.apply() - before: rexxInterface.jniRexxStart (...)");

        lastScriptCode=oscript.toString();   // save script's code, fall 2002, ---rgf

        if (vArgs==null || vArgs.isEmpty()) // no arguments supplied
        {
           return rexxInterface.jniRexxStart (lastScriptCode, null);
        }

        // 2003-01-06, ---rgf, deal with arguments (non-primitive Java objects) which need to be turned into Beans
        Vector beanVector = new Vector(vArgs.size());   // maximum size
        Object [] res;                  // to contain the results of args2RexxString()


        // this works under JRE 1.1.8, i.e. under older versions of OS/2 Warp as well, 2001-05-23, ---rgf
        String stringArg[] = new String [vArgs.size()];
        int i=0;
        Enumeration e=vArgs.elements();
        for ( i=0; e.hasMoreElements() ; i++)
        {
           Object ne=e.nextElement();
           if (ne!=null)
           {
               res = args2RexxString( ne );             // get String or Bean for this argument
               if (((Boolean) res[0]).booleanValue())   // a Bean had to be created
               {
                  beanVector.addElement(res[1]);   // remember beanName for later deregistering
               }
               stringArg[i]=(String) res[1];            // String value for Rexx
           }
           else
              stringArg[i]=null;
        }

if (bDebug)  System.err.println("RexxEngine.apply() - before: rexxInterface.jniRexxStart (...)");

        Object obj = rexxInterface.jniRexxStart (lastScriptCode, stringArg);

if (bDebug)  System.err.println("RexxEngine.apply() - after: rexxInterface.jniRexxStart (...)");

        // now deregister Beans
        e=beanVector.elements();        // get elements for Enumeration

        while ( e.hasMoreElements() )
        {
            rexxInterface.unregisterBean( (String) e.nextElement());
        }

if (bDebug)  System.err.println("RexxEngine.apply() - before returning ["+obj+"] (...)");
        return obj;
    }


    /** Turns received object either into a String, if object is a primitive type
      * or a Java wrapper class for primitive types. Otherwise it makes sure that
      * a Bean will be created for it and returns the beanName instead (a String).
      * <code>null</code> will get returned as <code>null</null>, hence no String
      * representation is returned instead; this way the Rexx <code>ARG()</code>
      * function can be used to determine, if such an argument was omitted or not.
      * (2003-01-06, ---rgf).
      *
      * @param o object to be turned into a String or a Bean.
      * @return a two-dimensional array of Object. The element with index 0 indicates whether
      * a Bean had to be created (<code>Boolean.TRUE</code> or <code>Boolean.FALSE</code>),
      * the second element with index 1 represents the object as a String or <code>null</code>
      * if the received object is <code>null</code>.
      */
    private Object [] args2RexxString(Object o)
    {
        Object res [] = { Boolean.FALSE, // use Boolean object representing false
                          o };

        if (o==null) return res;    // leave it as is

        Class cl = o.getClass();    // get class object

        if (cl==String.class)
        {
            return res;
        }

        else if (cl==Boolean.class )
        {
           res [1] = ( ((Boolean) o).booleanValue() ? "1" : "0");     // return boolean value
           return res;
        }


        else if ( cl.isPrimitive()     ||
                  cl==Integer.class    || cl==Character.class  ||
                  cl==Long.class       ||
                  cl==Float.class      || cl==Double.class     ||
                  cl==Byte.class       || cl==Short.class
                )
        {
            res[1]=o.toString();    // make sure a String object will get returned
            return res;
        }

        // not of a primitive type nor String, hence a bean will be created for it
        res[0] = Boolean.TRUE;          // a bean will be created for this
        res[1] = rexxInterface.makeString4Rexx( o );  // create Bean, return beanName (proxy for received Java object)
        return res;
    }




    /** Method implemented in order to avoid an exeption to be thrown by the
     *  default implementation.
      *
      * @param bean the bean to be declared for the already started language.
      */
    public void declareBean(BSFDeclaredBean bean)
                 throws BSFException
    {
        return;
    }

    /** Method implemented in order to avoid an exeption to be thrown by the
     *  default implementation.
      *
      * @param bean the bean to be declared for the already started language.
      */

    public void undeclareBean(BSFDeclaredBean bean)
                       throws BSFException
    {
        return;
    }
}
